<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CorsoResource\Pages;
use App\Models\Corso;
use App\Models\PropostaCorso;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use App\Models\CategoriaCorsi;
use BezhanSalleh\FilamentShield\Contracts\HasShieldPermissions;

class CorsoResource extends Resource implements HasShieldPermissions
{
    protected static ?string $model = Corso::class;

    protected static ?string $navigationIcon = 'heroicon-o-academic-cap';
    
    protected static ?string $navigationGroup = 'Gestione Corsi';
    
    protected static ?string $navigationLabel = 'Gestione Corsi';

    public static function getModelLabel(): string
    {
        return 'Corso';
    }

    public static function getPluralModelLabel(): string
    {
        return 'Corsi';
    }

    public static function getNavigationLabel(): string
    {
        return __('Gestione Corsi');
    }

    public static function getNavigationGroup(): ?string
    {
        return __('Gestione Corsi');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Informazioni Generali')
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                Select::make('categoria_id')
                                    ->label('Categoria')
                                    ->options(function () {
                                        // Ottieni le categorie dai corsi esistenti
                                        $categorieCorsi = CategoriaCorsi::whereIn('nome', [
                                            'CORSI DI LAUREA',
                                            'MASTER',
                                            'CORSI DI PERFEZIONAMENTO',
                                            'CORSI DI FORMAZIONE'
                                        ])->pluck('nome', 'id');

                                        // Ottieni le categorie dalle proposte corsi approvate
                                        $categorieProposte = \App\Models\PropostaCorso::where('stato', 'approvata')
                                            ->pluck('tipologia_percorso')
                                            ->unique()
                                            ->map(function ($tipologia) {
                                                return CategoriaCorsi::firstOrCreate(
                                                    ['nome' => $tipologia],
                                                    ['descrizione' => $tipologia]
                                                );
                                            })
                                            ->pluck('nome', 'id');

                                        // Unisci le collezioni
                                        return $categorieCorsi->union($categorieProposte);
                                    })
                                    ->required()
                                    ->searchable(),
                                TextInput::make('titolo')
                                    ->required()
                                    ->maxLength(255),
                            ]),
                        Grid::make(3)
                            ->schema([
                                TextInput::make('codice')
                                    ->maxLength(255),
                                TextInput::make('anno_accademico')
                                    ->maxLength(255),
                                TextInput::make('semestre')
                                    ->maxLength(255),
                            ]),
                    ]),
        
                Section::make('Dettagli del Corso')
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                TextInput::make('corso_di_laurea')
                                    ->maxLength(255),
                                TextInput::make('ssd')
                                    ->label('Settore Scientifico Disciplinare')
                                    ->maxLength(255),
                            ]),
                        Grid::make(2)
                            ->schema([
                                TextInput::make('cfu')
                                    ->numeric()
                                    ->label('CFU'),
                                TextInput::make('durata_anni')
                                    ->numeric()
                                    ->label('Durata (anni)'),
                            ]),
                        TextInput::make('presidente_cds')
                            ->label('Presidente CDS')
                            ->maxLength(255),
                        TextInput::make('titolo_accesso')
                            ->label('Titolo di Accesso')
                            ->maxLength(255),
                    ]),
        
                Section::make('Informazioni Economiche')
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                TextInput::make('costo')
                                    ->label('Costo')
                                    ->numeric()
                                    ->required()
                                    ->prefix('€'),
                                Toggle::make('rateizzabile')
                                    ->label('Rateizzabile')
                                    ->default(true),
                            ]),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('categoria.nome')
                    ->sortable()
                    ->searchable()
                    ->label('Categoria'),
                TextColumn::make('titolo')
                    ->sortable()
                    ->searchable()
                    ->limit(30)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        if (strlen($state) <= $column->getCharacterLimit()) {
                            return null;
                        }
                        return $state;
                    }),
                TextColumn::make('codice')
                    ->sortable()
                    ->searchable(),
                TextColumn::make('anno_accademico')
                    ->sortable(),
                TextColumn::make('cfu')
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCorsi::route('/'),
            'create' => Pages\CreateCorso::route('/create'),
            'edit' => Pages\EditCorso::route('/{record}/edit'),
        ];
    }

    public static function getPermissionPrefixes(): array
    {
        return [
            'view',
            'view_any',
            'create',
            'update',
            'delete',
            'delete_any'
        ];
    }
} 