<?php

namespace App\Filament\Resources\ProjectResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\Summarizers\Sum;
use Filament\Tables\Grouping\Group;
use Filament\Notifications\Notification;
use Filament\Support\Exceptions\Halt;
use Illuminate\Support\Facades\DB;

//SI PUO FARE MEGLIO?
use App\Models\BudgetSubcategory;
use App\Models\AnagraficheProject;
use App\Models\Anagrafica;

class BudgetItemsRelationManager extends RelationManager
{
    protected static string $relationship = 'budgetItems';

    protected static ?string $title = 'Piano Finanziario';

    protected static ?string $modelLabel = 'Voce di Budget';
    
    protected static ?string $pluralModelLabel = 'Voci di Budget';

    // SI PUO FARE MEGLIO? LO GUARDO A MANO DAL DB
    private int $personale_id = 1;


    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Dettagli Voce di Budget')
                    ->description('Inserisci i dettagli della voce di budget')
                    ->schema([
                        Forms\Components\Select::make('work_package_id')
                            ->relationship(
                                'workPackage',
                                'title',
                                fn ($query, $livewire) => $query
                                    ->where('project_id', $livewire->ownerRecord->id)
                                    ->orderBy('code')
                            )
                            ->getOptionLabelFromRecordUsing(fn ($record) => "{$record->code} - {$record->title}")
                            ->label('Work Package')
                            ->required()
                            ->preload()
                            ->searchable(),
                        Forms\Components\Select::make('budget_subcategory_id')
                            ->relationship('subcategory', 'name')
                            ->label('Contabilità Analitica')
                            ->required()
                            ->preload()
                            ->searchable()
                            ->live()
                            ->columnspan('full'),
                        Forms\Components\Grid::make(2)
                            ->schema([
                            Forms\Components\TextInput::make('gross_amount')
                                ->label('Importo Lordo')
                                ->required()
                                ->numeric()
                                ->prefix('€')
                                ->minValue(0),
                            Forms\Components\TextInput::make('reported_amount')
                                ->label('Importo Rendicontato')
                                ->required()
                                ->default(0)
                                ->numeric()
                                ->prefix('€')
                                ->minValue(0),
                            ])
                            ->visible(fn (Get $get) => $this->hideOrDisplayPeopleBudget($get, fn($a,$b) => $a !== $b)),
                        Forms\Components\Group::make()
                            ->schema([
                                Forms\Components\Select::make('anagrafica_id')
                                    ->label('Personale')
                                    ->options(Anagrafica::all()->pluck('nome_completo', 'id'))
                                    ->required()
                                    ->preload()
                                    ->searchable()
                                    ->columnspan('full'),
                                Forms\Components\Toggle::make('is_esterno')
                                    ->label('Interno/Esterno'),
                            ])
                            ->visible(fn (Get $get) => $this->hideOrDisplayPeopleBudget($get, fn($a,$b) => $a === $b))
                            ->columnspan('full'),
                        Forms\Components\DatePicker::make('start_date')
                            ->label('Dal')
                            ->required(),
                        Forms\Components\DatePicker::make('end_date')
                            ->label('Al')
                            ->required(),
                        Forms\Components\Textarea::make('notes')
                            ->label('Note')
                            ->maxLength(65535)
                            ->columnSpanFull(),
                    ])
                    ->columns(2),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('description')
            ->columns([
                Tables\Columns\TextColumn::make('workPackage.code')
                    ->label('WP')
                    ->sortable()
                    ->searchable(),
                Tables\Columns\TextColumn::make('subcategory.category.name')
                    ->label('Macro Voce')
                    ->sortable()
                    ->limit(20)
                    ->searchable(),
                Tables\Columns\TextColumn::make('subcategory.name')
                    ->label('Contabilità Analitica')
                    ->sortable()
                    ->limit(20)
                    ->searchable(),
                Tables\Columns\TextColumn::make('subcategory.code')
                    ->label('Codice')
                    ->searchable(),
                Tables\Columns\TextColumn::make('start_date')
                    ->label('Dal')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('end_date')
                    ->label('Al')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('gross_amount')
                    ->label('Importo Lordo')
                    ->money('EUR')
                    ->sortable()
                    ->summarize(
                        Sum::make()
                            ->label('Totale')
                            ->money('EUR')
                    ),
                Tables\Columns\TextColumn::make('reported_amount')
                    ->label('Importo Rendicontato')
                    ->money('EUR')
                    ->sortable()
                    ->summarize(
                        Sum::make()
                            ->label('Totale')
                            ->money('EUR')
                    ),
                Tables\Columns\TextColumn::make('to_report_amount')
                    ->label('Da Rendicontare')
                    ->money('EUR')
                    ->sortable()
                    ->summarize(
                        Sum::make()
                            ->label('Totale')
                            ->money('EUR')
                    ),
            ])
            ->groups([
                Tables\Grouping\Group::make('workPackage.code')
                    ->label('Work Package')
                    ->getTitleFromRecordUsing(function ($record): string {
                        $wp = $record->workPackage;
                        
                        // Calcola l'importo totale da rendicontare per il WP
                        $reportedTotal = $wp->budgetItems()->sum('reported_amount');
                        $toReportTotal = $wp->gross_amount - $reportedTotal;

                        // Costruisci il titolo del gruppo
                        $title = "Work Package: {$wp->code}" . 
                                " (Budget Totale: € " . number_format($wp->gross_amount, 2, ',', '.') . 
                                " - Da Rendicontare: € " . number_format($toReportTotal, 2, ',', '.') . ")";
                        
                        // Aggiungi il pallino verde solo se l'importo da rendicontare è 0
                        if ($toReportTotal == 0) {
                            $title .= " 🟢";
                        } else {
                            $title .= " 🔴";
                        }
                        
                        return $title;
                    })
                    ->collapsible()
            ])
            ->defaultGroup('workPackage.code')
            ->filters([
                Tables\Filters\SelectFilter::make('work_package_id')
                    ->relationship(
                        'workPackage',
                        'title',
                        fn ($query, $livewire) => $query
                            ->where('project_id', $livewire->ownerRecord->id)
                            ->orderBy('code')
                    )
                    ->getOptionLabelFromRecordUsing(fn ($record) => "{$record->code} - {$record->title}")
                    ->label('Work Package')
                    ->preload()
                    ->searchable(),
                Tables\Filters\SelectFilter::make('budget_subcategory_id')
                    ->relationship('subcategory.category', 'name')
                    ->label('Macro Voce')
                    ->preload()
                    ->searchable(),
                Tables\Filters\TernaryFilter::make('is_fully_reported')
                    ->label('Stato Rendicontazione')
                    ->placeholder('Tutti')
                    ->trueLabel('Completamente Rendicontati')
                    ->falseLabel('Da Rendicontare')
                    ->queries(
                        true: fn ($query) => $query->whereRaw('gross_amount = reported_amount'),
                        false: fn ($query) => $query->whereRaw('gross_amount > reported_amount'),
                    ),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->label('Nuova Voce')
                    ->before(function (array $data) {
                        // Simula il check del budget prima della creazione

                        //FT
                        //Solo se e` stato settato il "gross_amount"
                        //TODO
                        //Bisogna fare controlli sull'importo della risorsa umana? Da contratto o da altro? (quando si crea - ovviamente per il timesheet andranno fatti in altra sede)
                        if (isset($data['gross_amount'])) {
                            $budgetItem = new \App\Models\BudgetItem($data);
                            $workPackage = \App\Models\WorkPackage::find($data['work_package_id']);
                            $budgetItem->workPackage()->associate($workPackage);
                            
                            $currentTotal = $workPackage->budgetItems()->sum('gross_amount');
                            $remainingBudget = $workPackage->gross_amount - $currentTotal;
                            
                            if ($data['gross_amount'] > $remainingBudget) {
                                Notification::make()
                                    ->danger()
                                    ->title('Errore Budget')
                                    ->body("L'importo lordo (€ " . number_format($data['gross_amount'], 2, ',', '.') . ") supera il budget rimanente del WP (€ " . number_format($remainingBudget, 2, ',', '.') . ")")
                                    ->send();
                                    
                                throw new Halt();
                            }
                        }
                    })
                    ->after(function ($data, $record, $livewire) {
                        if (isset($data['anagrafica_id'])) {
                            $personale_record = [
                                'anagrafica_id' => $data['anagrafica_id'],
                                'work_package_id' => $data['work_package_id'],
                                'is_esterno' => $data['is_esterno'],
                                'budget_item_id' => $record->id,
                                'project_id' => $livewire->ownerRecord->id,
                            ];
                            AnagraficheProject::create($personale_record);
                        }
                        $this->dispatch('budgetItems::created');
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->label('Modifica')
                    ->after(function ($data, $record, $livewire) {
                        //Aggiorna tabella anagrafiche_project
                        $record->anagrafica_id = $data['anagrafica_id'];
                        $record->work_package_id = $data['anagrafica_id'];
                        $record->is_esterno = $data['is_esterno'];
                        AnagraficheProject::update($record);

                        $this->dispatch('budgetItems::updated');
                    }),
                Tables\Actions\DeleteAction::make()
                    ->label('Elimina')
                    ->before(function ($record) {
                        AnagraficheProject::where('budget_item_id', $record->id)->delete();
                    })
                    ->after(function () {
                        $this->dispatch('budgetItems::deleted');
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->label('Elimina Selezionati')
                        ->before(function ($records) {
                            foreach ($records as $record) {
                                AnagraficheProject::where('budget_item_id', $record->id)->delete();
                            }
                        })
                        ->after(function () {
                            $this->dispatch('budgetItems::deleted');
                        }),
                ]),
            ]);
    }

    private function hideOrDisplayPeopleBudget (Get $get, $compare) {
        $budget_category_item = BudgetSubcategory::where('id', $get('budget_subcategory_id'))->pluck('budget_category_id')->toArray();
        if (array_key_exists(0, $budget_category_item)) {
            return $compare($budget_category_item[0], $this->personale_id);
        } else {
            return false;
        }
    }

} 
