<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PropostaCorsoResource\Pages;
use App\Filament\Resources\PropostaCorsoResource\RelationManagers;
use App\Models\PropostaCorso;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\MarkdownEditor;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Table;
use Filament\Notifications\Notification;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use BezhanSalleh\FilamentShield\Contracts\HasShieldPermissions;
use Filament\Tables\Actions\Action;
use App\Http\Controllers\PropostaCorsoController;
use Filament\Forms\Components\Textarea as FilamentTextarea;
use Filament\Forms\Components\Forms;
use Illuminate\Support\Facades\Auth;
use Filament\Forms\Components\DatePicker;
use App\Models\CategoriaCorsi;
use App\Models\Corso;
use Filament\Facades\Filament;

class PropostaCorsoResource extends Resource implements HasShieldPermissions
{
    protected static ?string $model = PropostaCorso::class;

    protected static ?string $navigationIcon = 'heroicon-o-academic-cap';
    
    protected static ?string $navigationGroup = 'Gestione Corsi';
    
    protected static ?int $navigationSort = 1;

    protected static ?string $navigationLabel = 'Proposta corsi';

    protected static ?string $modelLabel = 'Proposta corso';
    
    protected static ?string $pluralModelLabel = 'Proposta corsi';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Toggle::make('is_completed')
                    ->label('Proposta completata')
                    ->helperText('Attiva questo toggle quando hai completato l\'inserimento di tutti i dati')
                    ->onColor('success')
                    ->offColor('danger')
                    ->columnSpanFull()
                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                    
                Section::make('Informazioni Generali')
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                TextInput::make('titolo')
                                    ->required()
                                    ->maxLength(255)
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                                Select::make('tipologia_percorso')
                                    ->required()
                                    ->options([
                                        'MASTER' => 'Master',
                                        'CORSI DI PERFEZIONAMENTO' => 'Corsi di Perfezionamento',
                                        'CORSI DI FORMAZIONE' => 'Corsi di Formazione',
                                    ])
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                            ]),
                        TextInput::make('titolo_esteso')
                            ->required()
                            ->maxLength(255)
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                        Grid::make(3)
                            ->schema([
                                TextInput::make('cfu')
                                    ->required()
                                    ->numeric()
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                                TextInput::make('modalita_erogazione')
                                    ->required()
                                    ->maxLength(255)
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                                TextInput::make('anno_accademico')
                                    ->required()
                                    ->maxLength(255)
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                            ]),
                        Grid::make(2)
                            ->schema([
                                DatePicker::make('data_inizio')
                                    ->label('Data inizio corso')
                                    ->format('d/m/Y')
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                                DatePicker::make('data_fine')
                                    ->label('Data fine corso')
                                    ->format('d/m/Y')
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza')
                                    ->after('data_inizio'),
                            ]),
                        Grid::make(2)
                            ->schema([
                                TextInput::make('direttore_corso')
                                    ->required()
                                    ->maxLength(255)
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                                TextInput::make('coordinatore_corso')
                                    ->required()
                                    ->maxLength(255)
                                    ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                            ]),
                    ]),

                Section::make('Dettagli del Corso')
                    ->schema([
                        MarkdownEditor::make('obiettivi_finalita')
                            ->label('Obiettivi e finalità')
                            ->required()
                            ->columnSpanFull()
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                        MarkdownEditor::make('destinatari')
                            ->required()
                            ->columnSpanFull()
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                        MarkdownEditor::make('durata')
                            ->required()
                            ->columnSpanFull()
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                        MarkdownEditor::make('coerenza_metodologia')
                            ->label('Coerenza della metodologia didattica con modello IUL')
                            ->required()
                            ->columnSpanFull()
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                        MarkdownEditor::make('modalita_conseguimento')
                            ->label('Modalità di conseguimento titolo')
                            ->required()
                            ->columnSpanFull()
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                        MarkdownEditor::make('costo_iscrizione')
                            ->required()
                            ->columnSpanFull()
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                        MarkdownEditor::make('partner_scientifici')
                            ->label('Eventuali partner scientifici')
                            ->columnSpanFull()
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                        MarkdownEditor::make('partner_organizzativi')
                            ->label('Eventuali partner organizzativi')
                            ->columnSpanFull()
                            ->disabled(fn ($record) => $record?->stato && $record->stato !== 'bozza'),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('titolo')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('tipologia_percorso')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('anno_accademico')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('data_inizio')
                    ->label('Data inizio')
                    ->date('d/m/Y')
                    ->sortable(),
                TextColumn::make('data_fine')
                    ->label('Data fine')
                    ->date('d/m/Y')
                    ->sortable(),
                Tables\Columns\IconColumn::make('is_completed')
                    ->label('Completata')
                    ->boolean()
                    ->sortable(),
                TextColumn::make('stato')
                    ->badge()
                    ->label('Stato proposta')
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'bozza' => 'In lavorazione',
                        'inviata' => 'Inviata per approvazione',
                        'approvata' => 'Approvata',
                        'rifiutata' => 'Rifiutata',
                    })
                    ->color(fn (string $state): string => match ($state) {
                        'bozza' => 'gray',
                        'inviata' => 'warning',
                        'approvata' => 'success',
                        'rifiutata' => 'danger',
                    })
                    ->searchable()
                    ->sortable(),
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('data_approvazione')
                    ->label('Data approvazione')
                    ->date()
                    ->sortable()
                    ->visible(fn ($record) => $record?->stato === 'approvata'),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn ($record) => $record?->stato === 'bozza'),
                Tables\Actions\Action::make('download_pdf')
                    ->label('Scarica PDF')
                    ->icon('heroicon-o-document-arrow-down')
                    ->color('success')
                    ->url(fn ($record) => route('proposta-corso.pdf', $record))
                    ->openUrlInNewTab()
                    ->visible(fn ($record) => $record?->stato !== 'bozza'),
                Tables\Actions\Action::make('invia')
                    ->label('Invia per approvazione')
                    ->icon('heroicon-o-paper-airplane')
                    ->color('warning')
                    ->requiresConfirmation()
                    ->visible(fn ($record) => $record?->stato === 'bozza' && $record?->is_completed)
                    ->action(function ($record) {
                        app(PropostaCorsoController::class)->inviaPerApprovazione($record);
                    })
                    ->successNotification(
                        Notification::make()
                            ->success()
                            ->title('Proposta inviata')
                            ->body('La proposta è stata inviata per approvazione alla segreteria organi.'),
                    ),
                Tables\Actions\Action::make('approve')
                    ->label('Approva')
                    ->icon('heroicon-o-check-circle')
                    ->color('success')
                    ->requiresConfirmation()
                    ->modalHeading('Approva proposta')
                    ->modalDescription('Sei sicuro di voler approvare questa proposta? Verrà automaticamente creato il corso.')
                    ->visible(fn ($record) => 
                        $record?->stato === 'inviata' && 
                        Filament::auth()->user()->hasRole(['super_admin', 'segreteria_organi'])
                    )
                    ->action(function ($record) {
                        $record->update([
                            'stato' => 'approvata',
                            'data_approvazione' => now(),
                        ]);

                        // Crea la categoria se non esiste
                        $categoria = CategoriaCorsi::firstOrCreate(
                            ['nome' => $record->tipologia_percorso],
                            ['descrizione' => $record->tipologia_percorso]
                        );

                        // Crea il corso
                        Corso::create([
                            'categoria_id' => $categoria->id,
                            'titolo' => $record->titolo,
                            'anno_accademico' => $record->anno_accademico,
                            'cfu' => $record->cfu,
                            'durata_anni' => 1,
                            'presidente_cds' => $record->direttore_corso,
                            'titolo_accesso' => '',
                            'costo' => 0,
                            'rateizzabile' => true,
                        ]);

                        Notification::make()
                            ->success()
                            ->title('Proposta approvata')
                            ->body('La proposta è stata approvata e il corso è stato creato con successo.')
                            ->send();
                    }),
                Tables\Actions\Action::make('reject')
                    ->label('Rifiuta')
                    ->icon('heroicon-o-x-circle')
                    ->color('danger')
                    ->requiresConfirmation()
                    ->modalHeading('Rifiuta proposta')
                    ->modalDescription('Sei sicuro di voler rifiutare questa proposta?')
                    ->form([
                        Textarea::make('motivo_rifiuto')
                            ->label('Motivo del rifiuto')
                            ->required()
                            ->maxLength(1000),
                    ])
                    ->visible(fn ($record) => 
                        $record?->stato === 'inviata' && 
                        Filament::auth()->user()->hasRole(['super_admin', 'segreteria_organi'])
                    )
                    ->action(function ($record, array $data) {
                        $record->update([
                            'stato' => 'rifiutata',
                            'motivo_rifiuto' => $data['motivo_rifiuto'],
                        ]);

                        Notification::make()
                            ->success()
                            ->title('Proposta rifiutata')
                            ->body('La proposta è stata rifiutata.')
                            ->send();
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            RelationManagers\ModuliRelationManager::class,
            RelationManagers\FigureProfessionaliRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPropostaCorsos::route('/'),
            'create' => Pages\CreatePropostaCorso::route('/create'),
            'edit' => Pages\EditPropostaCorso::route('/{record}/edit'),
        ];
    }

    public static function getPermissionPrefixes(): array
    {
        return [
            'view',
            'view_any',
            'create',
            'update',
            'delete',
            'delete_any',
            'approve',
        ];
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::where('stato', 'bozza')->count();
    }

    public static function getNavigationGroup(): ?string
    {
        return __('Gestione Corsi');
    }

    public static function getNavigationLabel(): string
    {
        return __('Proposta Corsi');
    }

    public static function getPluralModelLabel(): string
    {
        return __('Proposte Corsi');
    }

    public static function getModelLabel(): string
    {
        return __('Proposta Corso');
    }
}
