<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TipologiaUtenteResource\Pages;
use App\Models\TipologiaUtente;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Str;

class TipologiaUtenteResource extends Resource
{
    protected static ?string $model = TipologiaUtente::class;

    protected static ?string $navigationIcon = 'heroicon-o-user-group';
    protected static ?string $navigationGroup = 'Anagrafica';
    protected static ?int $navigationSort = 2;
    protected static ?string $modelLabel = 'Tipologia Utente';
    protected static ?string $pluralModelLabel = 'Tipologie';
    protected static ?string $navigationLabel = 'Tipologie Anagrafiche';
    public $pageTitle = 'Tipologie Anagrafiche';


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Grid::make()
                    ->schema([
                        Forms\Components\Section::make('Informazioni Generali')
                            ->description('Informazioni base della tipologia utente')
                            ->icon('heroicon-o-user')
                            ->schema([
                                Forms\Components\Select::make('tipo_utente')
                                    ->label('Tipo Utente')
                                    ->options([
                                        'strutturato' => 'Personale strutturato',
                                        'docente_contratto' => 'Personale docente a contratto',
                                        'tutor' => 'Personale tutor didattici',
                                        'tecnico' => 'Personale Tecnico Amministrativo',
                                        'occasionale' => 'Collaboratori occasionali',
                                        'cococo' => 'Collaboratori cococo',
                                        'piva' => 'Consulenti p.iva'
                                    ])
                                    ->required()
                                    ->live()
                                    ->afterStateHydrated(function (string|null $state, Forms\Components\Select $component) {
                                        if ($state === null) {
                                            $record = $component->getRecord();
                                            if ($record) {
                                                $nome = strtolower($record->nome);
                                                if (str_contains($nome, 'strutturato')) {
                                                    $component->state('strutturato');
                                                } elseif (str_contains($nome, 'docente a contratto')) {
                                                    $component->state('docente_contratto');
                                                } elseif (str_contains($nome, 'tutor')) {
                                                    $component->state('tutor');
                                                } elseif (str_contains($nome, 'tecnico')) {
                                                    $component->state('tecnico');
                                                } elseif (str_contains($nome, 'occasionali')) {
                                                    $component->state('occasionale');
                                                } elseif (str_contains($nome, 'cococo')) {
                                                    $component->state('cococo');
                                                } elseif (str_contains($nome, 'p.iva')) {
                                                    $component->state('piva');
                                                }
                                            }
                                        }
                                    })
                                    ->columnSpanFull(),
                                Forms\Components\Grid::make()
                                    ->schema([
                                        Forms\Components\TextInput::make('nome')
                                            ->required()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function ($state, callable $set) {
                                                if ($state) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            })
                                            ->columnSpan(2),
                                        Forms\Components\TextInput::make('slug')
                                            ->required()
                                            ->maxLength(255)
                                            ->unique(ignoreRecord: true)
                                            ->disabled()
                                            ->dehydrated(),
                                    ])
                                    ->columns(3),
                                Forms\Components\Textarea::make('descrizione')
                                    ->maxLength(65535)
                                    ->columnSpanFull(),
                            ])
                            ->columns(1)
                            ->columnSpan(['lg' => 2]),

                        Forms\Components\Section::make('Note Aggiuntive')
                            ->icon('heroicon-o-pencil-square')
                            ->schema([
                                Forms\Components\Textarea::make('regole.note')
                                    ->label('Note')
                                    ->rows(4)
                                    ->columnSpanFull(),
                            ])
                            ->columnSpan(['lg' => 1])
                            ->collapsible(),
                    ])
                    ->columns(['lg' => 3]),

                Forms\Components\Section::make('Regole Personale Docente')
                    ->description('Configurazione delle regole per i professori')
                    ->icon('heroicon-o-academic-cap')
                    ->schema([
                        Forms\Components\Repeater::make('regole.professori')
                            ->label('Regole Professori')
                            ->schema([
                                Forms\Components\Select::make('tipo')
                                    ->label('Tipo')
                                    ->options([
                                        'tempo_pieno' => 'Tempo Pieno',
                                        'tempo_definito' => 'Tempo Definito'
                                    ])
                                    ->required()
                                    ->columnSpan(1),
                                Forms\Components\Grid::make()
                                    ->schema([
                                        Forms\Components\TextInput::make('ore_anno')
                                            ->label('Ore Anno')
                                            ->numeric()
                                            ->required(),
                                        Forms\Components\TextInput::make('ore_giorno')
                                            ->label('Ore Giorno')
                                            ->numeric()
                                            ->required(),
                                        Forms\Components\TextInput::make('giorni_mese')
                                            ->label('Giorni Mese')
                                            ->numeric()
                                            ->required(),
                                        Forms\Components\TextInput::make('ore_mese')
                                            ->label('Ore Mese')
                                            ->numeric()
                                            ->required(),
                                        Forms\Components\TextInput::make('tempo_produttivo')
                                            ->label('Tempo Produttivo')
                                            ->numeric()
                                            ->required(),
                                    ])
                                    ->columns(3)
                                    ->columnSpan(2),
                            ])
                            ->columns(3)
                            ->defaultItems(0)
                            ->addActionLabel('Aggiungi regola professori')
                            ->visible(fn ($get) => $get('tipo_utente') === 'strutturato')
                            ->afterStateHydrated(function ($state, Forms\Components\Repeater $component) {
                                if (empty($state)) {
                                    $component->state([
                                        [
                                            'tipo' => 'tempo_pieno',
                                            'ore_anno' => 1500,
                                            'ore_giorno' => 6,
                                            'giorni_mese' => 22,
                                            'ore_mese' => 125,
                                            'tempo_produttivo' => 1150
                                        ],
                                        [
                                            'tipo' => 'tempo_definito',
                                            'ore_anno' => 750,
                                            'ore_giorno' => 3,
                                            'giorni_mese' => 22,
                                            'ore_mese' => 62.5,
                                            'tempo_produttivo' => 500
                                        ]
                                    ]);
                                }
                            }),
                    ])
                    ->collapsible()
                    ->visible(fn ($get) => $get('tipo_utente') === 'strutturato'),

                Forms\Components\Section::make('Regole Ricercatori')
                    ->description('Configurazione delle regole per i ricercatori')
                    ->icon('heroicon-o-academic-cap')
                    ->schema([
                        Forms\Components\Repeater::make('regole.ricercatori')
                            ->label('Regole Ricercatori')
                            ->schema([
                                Forms\Components\Select::make('tipo')
                                    ->label('Tipo')
                                    ->options([
                                        'tempo_pieno' => 'Tempo Pieno',
                                        'tempo_definito' => 'Tempo Definito'
                                    ])
                                    ->required()
                                    ->columnSpan(1),
                                Forms\Components\Grid::make()
                                    ->schema([
                                        Forms\Components\TextInput::make('ore_anno')
                                            ->label('Ore Anno')
                                            ->numeric()
                                            ->required(),
                                        Forms\Components\TextInput::make('ore_giorno')
                                            ->label('Ore Giorno')
                                            ->numeric()
                                            ->required(),
                                        Forms\Components\TextInput::make('giorni_mese')
                                            ->label('Giorni Mese')
                                            ->numeric()
                                            ->required(),
                                        Forms\Components\TextInput::make('ore_mese')
                                            ->label('Ore Mese')
                                            ->numeric()
                                            ->required(),
                                        Forms\Components\TextInput::make('tempo_produttivo')
                                            ->label('Tempo Produttivo')
                                            ->numeric()
                                            ->required(),
                                    ])
                                    ->columns(3)
                                    ->columnSpan(2),
                            ])
                            ->columns(3)
                            ->defaultItems(0)
                            ->addActionLabel('Aggiungi regola ricercatori')
                            ->visible(fn ($get) => $get('tipo_utente') === 'strutturato')
                            ->afterStateHydrated(function ($state, Forms\Components\Repeater $component) {
                                if (empty($state)) {
                                    $component->state([
                                        [
                                            'tipo' => 'tempo_pieno',
                                            'ore_anno' => 1500,
                                            'ore_giorno' => 6,
                                            'giorni_mese' => 22,
                                            'ore_mese' => 125,
                                            'tempo_produttivo' => 1150
                                        ],
                                        [
                                            'tipo' => 'tempo_definito',
                                            'ore_anno' => 750,
                                            'ore_giorno' => 3,
                                            'giorni_mese' => 22,
                                            'ore_mese' => 62.5,
                                            'tempo_produttivo' => 550
                                        ]
                                    ]);
                                }
                            }),
                    ])
                    ->collapsible()
                    ->visible(fn ($get) => $get('tipo_utente') === 'strutturato'),

                Forms\Components\Section::make('Regole Personale Tecnico Amministrativo')
                    ->description('Configurazione delle regole per il personale tecnico amministrativo')
                    ->icon('heroicon-o-briefcase')
                    ->schema([
                        Forms\Components\Grid::make()
                            ->schema([
                                Forms\Components\TextInput::make('regole.ore_anno')
                                    ->label('Ore Anno')
                                    ->numeric(),
                                Forms\Components\TextInput::make('regole.ore_giorno')
                                    ->label('Ore Giorno')
                                    ->numeric(),
                            ])
                            ->columns(2),
                    ])
                    ->collapsible()
                    ->visible(fn ($get) => $get('tipo_utente') === 'tecnico'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nome')
                    ->searchable(),
                Tables\Columns\TextColumn::make('slug')
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTipologieUtente::route('/'),
            'create' => Pages\CreateTipologiaUtente::route('/create'),
            'edit' => Pages\EditTipologiaUtente::route('/{record}/edit'),
        ];
    }
} 