<?php

namespace App\Livewire;

use App\Models\Anagrafica;
use App\Models\Contratto;
use App\Models\Corso;
use App\Models\Event;
use App\Models\TipoContratto;
use App\Models\ContrattoTemplate;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Toggle;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Components\Actions;
use Filament\Forms\Components\Actions\Action;
use Filament\Notifications\Notification;

use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Str;
use Filament\Forms\Components\ViewField;


class CreateContratto extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public Contratto $contratto;
    public $isEditing = false;
    public $selectedTipoContratto = null;
    public $selectedAnagrafica = null;
    public $nome_template = '';
    public $contenuto_custom = '';
    public $template_id = null;

    public function mount(Request $request): void
    {
        $id = $request->input('id');
        if ($id) {
            $this->contratto = Contratto::with(['corsi', 'template'])->findOrFail($id);
            $data = $this->contratto->toArray();
            
            // Imposta usa_template_custom a true se esiste un template_id o contenuto_custom
            $data['usa_template_custom'] = !empty($this->contratto->template_id) || !empty($this->contratto->contenuto_custom);
            
            // Se c'è un contenuto custom nel contratto, usa quello
            if (!empty($this->contratto->contenuto_custom)) {
                $data['contenuto_custom'] = $this->contratto->contenuto_custom;
            } 
            // Altrimenti, se c'è un template, usa il contenuto del template
            elseif ($this->contratto->template) {
                $data['contenuto_custom'] = $this->contratto->template->contenuto;
            }
            
            // Mantieni sempre il template_id se esiste
            $data['template_id'] = $this->contratto->template_id;
            
            // Imposta i corsi direttamente nel data array
            $data['corsi'] = $this->contratto->corsi->pluck('id')->toArray();
            
            $this->isEditing = true;
            $this->selectedTipoContratto = $this->contratto->tipo_contratto_id;
            $this->selectedAnagrafica = $this->contratto->anagrafica_id;

            // Assicurati che i dati vengano impostati correttamente
            $this->data = $data;
            $this->form->fill($data);
        } else {
            $this->contratto = new Contratto();
            $data = [];
            $this->data = $data;
            $this->form->fill($data);
        }
    }

    public function createAndSendContract()
    {
        $this->validate();
        $data = $this->form->getState();
    
        if ($this->isEditing) {
            $this->contratto->update($data);
        } else {
            $this->contratto = Contratto::create($data);
        }
    
        // Genera il PDF
        $pdf = $this->generatePdf();
    
        // Salva il PDF
        $fileName = 'contratto_' . $this->contratto->id . '_' . now()->format('Y-m-d_His') . '.pdf';
        $pdfPath = 'contratti/' . $fileName;
        Storage::disk('public')->put($pdfPath, $pdf);
    
        // Aggiorna il contratto con il percorso del PDF
        $this->contratto->update(['pdf_path' => $pdfPath]);
    
        // Genera l'URL per il download
        $downloadUrl = route('download.contratto.pdf', ['id' => $this->contratto->id]);
    
        \Filament\Notifications\Notification::make()
            ->title('Contratto creato e PDF generato')
            ->body('Clicca qui per scaricare il PDF')
            ->actions([
                \Filament\Notifications\Actions\Action::make('download')
                    ->label('Scarica PDF')
                    ->url($downloadUrl)
                    ->openUrlInNewTab(),
            ])
            ->success()
            ->send();
    }


    protected function generatePdf()
    {
        // Carica le relazioni necessarie
        $this->contratto->load(['anagrafica', 'tipoContratto', 'corso']);
        $anagrafica = $this->contratto->anagrafica;
        $tipoContratto = $this->contratto->tipoContratto;

        // Se usa un template custom, usa il contenuto personalizzato
        if ($this->data['usa_template_custom'] ?? false) {
            \Illuminate\Support\Facades\Log::info('Usando template personalizzato', [
                'template_id' => $this->data['template_id'] ?? null,
                'usa_template_custom' => $this->data['usa_template_custom'],
                'contenuto_custom' => $this->data['contenuto_custom'] ?? null
            ]);

            // Prepara le variabili per i placeholder
            $variables = [
                'nome' => $anagrafica->nome,
                'cognome' => $anagrafica->cognome,
                'indirizzo' => $anagrafica->indirizzo_residenza,
                'citta' => $anagrafica->citta_residenza,
                'cap' => $anagrafica->cap_residenza,
                'provincia' => $anagrafica->provincia_residenza,
                'codice_fiscale' => $anagrafica->codice_fiscale,
                'partita_iva' => $anagrafica->partita_iva,
                'luogo_nascita' => $anagrafica->luogo_nascita,
                'data_nascita' => $anagrafica->data_nascita ? $anagrafica->data_nascita->format('d/m/Y') : '',
                'data_oggi' => now()->format('d/m/Y'),
                'compenso' => number_format($this->contratto->compenso, 2, ',', '.'),
                'data_inizio' => $this->contratto->data_inizio->format('d/m/Y'),
                'data_fine' => $this->contratto->data_fine->format('d/m/Y'),
                'ore_totali' => $this->contratto->ore_totali,
                'corso' => $this->contratto->corso ? $this->contratto->corso->titolo : '',
                'ssd' => $this->contratto->corso ? $this->contratto->corso->ssd : '',
                'cfu' => $this->contratto->cfu,
                'semestre' => $this->contratto->semestre,
                'anno_accademico' => $this->contratto->anno_accademico,
                'descrizione_incarico' => $this->contratto->descrizione_incarico,
                'numero_decreto' => $this->contratto->numero_decreto,
                'numero_posizioni' => $this->contratto->numero_posizioni,
                'area' => $this->contratto->area,
                'macrosettore' => $this->contratto->macrosettore,
                'settore_concorsuale' => $this->contratto->settore_concorsuale,
                'data_delibera_dipartimento' => $this->contratto->data_delibera_dipartimento ? $this->contratto->data_delibera_dipartimento->format('d/m/Y') : '',
                'data_delibera_cda' => $this->contratto->data_delibera_cda ? $this->contratto->data_delibera_cda->format('d/m/Y') : '',
                'stipendio' => number_format($this->contratto->stipendio, 2, ',', '.'),
                'tredicesima' => number_format($this->contratto->tredicesima, 2, ',', '.'),
                'regime_tempo' => $this->contratto->regime_tempo,
                'tipo_tempo' => $this->contratto->tipo_tempo,
                'tipo_orario' => $this->contratto->tipo_orario,
                'profilo' => $this->contratto->profilo,
                'livello' => $this->contratto->livello,
                'ore_settimanali' => $this->contratto->ore_settimanali,
                'retribuzione_mensile' => number_format($this->contratto->retribuzione_mensile, 2, ',', '.'),
                'retribuzione_annua' => number_format($this->contratto->retribuzione_annua, 2, ',', '.'),
            ];

            \Illuminate\Support\Facades\Log::info('Variabili template:', [
                'variables' => $variables
            ]);

            // Ottieni il contenuto del template
            $content = $this->data['contenuto_custom'] ?? '';
            if (empty($content) && $this->data['template_id']) {
                $template = ContrattoTemplate::find($this->data['template_id']);
                if ($template) {
                    $content = $template->contenuto;
                }
            }

            // Sostituisci le variabili nel template
            foreach ($variables as $key => $value) {
                $content = str_replace('{{' . $key . '}}', $value ?? '', $content);
            }

            \Illuminate\Support\Facades\Log::info('Contenuto finale del template', [
                'content' => $content
            ]);

            return PDF::loadView('pdf.contratto-custom', ['content' => $content])
                ->setPaper('a4')
                ->output();
        }
        
        // Se il tipo contratto ha slug 'test', usa il template di test
        if ($tipoContratto->slug === 'test') {
            // Ottieni i dati del secondo firmatario dal database
            $secondoFirmatario = \App\Models\Firmatario::findOrFail($this->contratto->secondo_firmatario);
            
            $secondoFirmatarioInfo = [
                'first_name' => $secondoFirmatario->nome,
                'last_name' => $secondoFirmatario->cognome,
                'title' => $secondoFirmatario->titolo . ' - ' . $secondoFirmatario->ruolo
            ];

            return \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.contratto-test', [
                'secondoFirmatarioInfo' => $secondoFirmatarioInfo
            ])
                ->setPaper('a4')
                ->output();
        }

        // Mappa dei profili di contratto
        $templateMap = [
            1 => 'pdf.contratto-profilo-1',  // Personale strutturato
            2 => 'pdf.contratto-profilo-2',  // Docenza
            3 => 'pdf.contratto-profilo-3',  // Tutoraggio
            4 => 'pdf.contratto-profilo-4',  // Contratto di lavoro dipendente
            5 => 'pdf.contratto-profilo-5',  // Collaborazione occasionale
            6 => 'pdf.contratto-profilo-6',  // COCOCO
            7 => 'pdf.contratto-profilo-7',  // Collaborazione autonoma (P.IVA)
        ];

        // Se esiste un template specifico per questo tipo di contratto, usalo
        if (isset($templateMap[$tipoContratto->id])) {
            return \Barryvdh\DomPDF\Facade\Pdf::loadView($templateMap[$tipoContratto->id], [
                'contratto' => $this->contratto,
                'anagrafica' => $anagrafica,
                'tipoContratto' => $tipoContratto,
            ])
                ->setPaper('a4')
                ->output();
        }

        $corso = $this->contratto->corso;

        // Prepara le variabili per i placeholder
        $variables = [
            'nome' => $anagrafica->nome,
            'cognome' => $anagrafica->cognome,
            'indirizzo' => $anagrafica->indirizzo_residenza,
            'citta' => $anagrafica->citta_residenza,
            'cap' => $anagrafica->cap_residenza,
            'provincia' => $anagrafica->provincia_residenza,
            'codice_fiscale' => $anagrafica->codice_fiscale,
            'partita_iva' => $anagrafica->partita_iva,
            'luogo_nascita' => $anagrafica->luogo_nascita,
            'data_nascita' => $anagrafica->data_nascita ? $anagrafica->data_nascita->format('d/m/Y') : '',
            'data_oggi' => now()->format('d/m/Y'),
            'compenso' => number_format($this->contratto->compenso, 2, ',', '.'),
            'data_inizio' => $this->contratto->data_inizio->format('d/m/Y'),
            'data_fine' => $this->contratto->data_fine->format('d/m/Y'),
            'ore_totali' => $this->contratto->ore_totali,
            'corso' => $corso ? $corso->titolo : '',
            'ssd' => $corso ? $corso->ssd : '',
            'cfu' => $this->contratto->cfu,
            'semestre' => $this->contratto->semestre,
            'anno' => $this->contratto->anno,
            'classe' => $this->contratto->classe,
            'anno_accademico' => $this->contratto->anno_accademico,
            'numero_decreto' => $this->contratto->numero_decreto,
            'numero_posizioni' => $this->contratto->numero_posizioni,
            'area' => $this->contratto->area,
            'macrosettore' => $this->contratto->macrosettore,
            'settore_concorsuale' => $this->contratto->settore_concorsuale,
            'data_delibera_dipartimento' => $this->contratto->data_delibera_dipartimento ? $this->contratto->data_delibera_dipartimento->format('d/m/Y') : '',
            'data_delibera_cda' => $this->contratto->data_delibera_cda ? $this->contratto->data_delibera_cda->format('d/m/Y') : '',
            'stipendio' => number_format($this->contratto->stipendio, 2, ',', '.'),
            'tredicesima' => number_format($this->contratto->tredicesima, 2, ',', '.'),
            'regime_tempo' => $this->contratto->regime_tempo,
            'tipo_tempo' => $this->contratto->tipo_tempo,
            'tipo_orario' => $this->contratto->tipo_orario,
            'profilo' => $this->contratto->profilo,
            'livello' => $this->contratto->livello,
            'ore_settimanali' => $this->contratto->ore_settimanali,
            'retribuzione_mensile' => number_format($this->contratto->retribuzione_mensile, 2, ',', '.'),
            'retribuzione_annua' => number_format($this->contratto->retribuzione_annua, 2, ',', '.'),
        ];

            $viewData = [
                'contratto' => $this->contratto,
                'anagrafica' => $anagrafica,
                'tipoContratto' => $tipoContratto,
            ];
            $view = 'pdf.contratto';

        $pdf = PDF::loadView($view, $viewData);
        return $pdf->output();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                \Filament\Forms\Components\Group::make()
                    ->schema([
                        \Filament\Forms\Components\Section::make('Informazioni Generali')
                    ->description('Seleziona l\'anagrafica e il tipo di contratto da creare')
                    ->schema([
                        Select::make('anagrafica_id')
                            ->label('Anagrafica')
                            ->searchable()
                            ->preload() 
                            ->options(function () {
                                return DB::table('anagrafiche')
                                    ->whereNull('deleted_at')
                                    ->where('is_complete', true)
                                    ->select('id', DB::raw("CONCAT(nome, ' ', cognome) AS nome_completo"))
                                    ->orderBy('nome')
                                    ->limit(100)
                                    ->pluck('nome_completo', 'id')
                                    ->toArray();
                            })
                            ->getSearchResultsUsing(function (string $search) {
                                return DB::table('anagrafiche')
                                    ->whereNull('deleted_at')
                                    ->where('is_complete', true)
                                    ->whereRaw("MATCH(nome, cognome) AGAINST(? IN BOOLEAN MODE)", [$search . '*'])
                                    ->select('id', DB::raw("CONCAT(nome, ' ', cognome) AS nome_completo"))
                                    ->limit(50)
                                    ->pluck('nome_completo', 'id')
                                    ->toArray();
                            })
                            ->getOptionLabelUsing(function ($value) {
                                $anagrafica = DB::table('anagrafiche')
                                    ->whereNull('deleted_at')
                                    ->where('id', $value)
                                    ->select(DB::raw("CONCAT(nome, ' ', cognome) AS nome_completo"))
                                    ->first();
                                return $anagrafica ? $anagrafica->nome_completo : null;
                            })
                            ->required()
                            ->loadingMessage('Caricamento anagrafiche...')
                            ->noSearchResultsMessage('Nessuna anagrafica trovata')
                            ->searchingMessage('Ricerca in corso...')
                            ->searchDebounce(300),
    
                        Select::make('tipo_contratto_id')
                            ->label('Tipo di Contratto')
                            ->options(TipoContratto::pluck('nome', 'id'))
                            ->required()
                                    ->live()
                            ->afterStateUpdated(fn ($state) => $this->selectedTipoContratto = $state),

                        Select::make('secondo_firmatario')
                            ->label('Secondo Firmatario')
                            ->options(function () {
                                return \App\Models\Firmatario::where('attivo', true)
                                    ->get()
                                    ->mapWithKeys(function ($firmatario) {
                                        return [$firmatario->id => "{$firmatario->titolo} {$firmatario->nome} {$firmatario->cognome} - {$firmatario->ruolo}"];
                                    });
                            })
                            ->required()
                            ->searchable()
                            ->preload()
                            ->helperText('Seleziona chi deve firmare il contratto dopo il primo firmatario'),

                                Grid::make(2)
                                    ->schema([
                                        DatePicker::make('data_inizio')
                                            ->label('Data Inizio')
                                            ->required(),

                                        DatePicker::make('data_fine')
                                            ->label('Data Fine')
                                            ->required(),
                                    ]),
                            ]),

                        \Filament\Forms\Components\Section::make('Informazioni Economiche')
                            ->schema([
                                Grid::make(3)
                                    ->schema([
                                        TextInput::make('compenso')
                                            ->label('Compenso')
                                            ->numeric()
                                            ->required()
                                            ->prefix('€')
                                            ->step(0.01)
                                            ->inputMode('decimal'),

                                        TextInput::make('costo_lordo_aziendale')
                                            ->label('Costo Lordo Aziendale')
                                            ->numeric()
                                            ->prefix('€'),

                                        TextInput::make('costo_lordo_dipendente')
                                            ->label('Costo Lordo Dipendente')
                                            ->numeric()
                                            ->prefix('€'),

                                        TextInput::make('costo_orario')
                                            ->label('Costo Orario')
                                            ->numeric()
                                            ->prefix('€'),

                                        TextInput::make('costo_giornaliero')
                                            ->label('Costo Giornaliero')
                                            ->numeric()
                                            ->prefix('€'),
                                    ]),
                            ])
                            ->collapsible(),

                        \Filament\Forms\Components\Section::make('Dettagli Contratto')
                                    ->schema([
                                        Select::make('corsi')
                                            ->label('Corsi')
                                            ->multiple()
                                            ->options(Corso::pluck('titolo', 'id'))
                                            ->searchable()
                                            ->preload()
                                    ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                Grid::make(2)
                                    ->schema([
                                        TextInput::make('numero_decreto')
                                            ->label('Numero Decreto Presidenziale')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        TextInput::make('numero_posizioni')
                                            ->label('Numero Posizioni')
                                            ->required()
                                            ->numeric()
                                            ->default(1)
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        Select::make('regime_tempo')
                                            ->label('Regime di Tempo')
                                            ->options([
                                                'pieno' => 'Tempo Pieno',
                                                'definito' => 'Tempo Definito'
                                            ])
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        TextInput::make('macrosettore')
                                            ->label('Macrosettore')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        TextInput::make('settore_concorsuale')
                                            ->label('Settore Concorsuale')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        TextInput::make('gsd')
                                            ->label('Gruppo Scientifico Disciplinare (GSD)')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        TextInput::make('ssd')
                                            ->label('Settore Scientifico Disciplinare (SSD)')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        TextInput::make('numero_decreto_dg')
                                            ->label('Numero Decreto DG')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        DatePicker::make('data_delibera_dipartimento')
                                            ->label('Data Delibera Dipartimento')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                        DatePicker::make('data_delibera_cda')
                                            ->label('Data Delibera CDA')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),
                                    ])
                                    ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),

                                Grid::make(2)
                                    ->schema([
                                        Select::make('tipo_tempo')
                                            ->label('Tipo Contratto')
                                            ->options([
                                                'determinato' => 'Determinato',
                                                'indeterminato' => 'Indeterminato'
                                            ])
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),

                                        Select::make('tipo_orario')
                                            ->label('Tipo Orario')
                                            ->options([
                                                'pieno' => 'Tempo Pieno',
                                                'parziale' => 'Part Time'
                                            ])
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),

                                        TextInput::make('profilo')
                                            ->label('Profilo Professionale')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),

                                        TextInput::make('mansioni')
                                            ->label('Mansioni')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),

                                        TextInput::make('livello')
                                            ->label('Livello CCNL')
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),

                                        TextInput::make('ore_settimanali')
                                            ->label('Ore Settimanali')
                                            ->numeric()
                                            ->default(40)
                                            ->required()
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),

                                        TextInput::make('retribuzione_mensile')
                                            ->label('Retribuzione Mensile')
                                            ->numeric()
                                            ->required()
                                            ->prefix('€')
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),

                                        TextInput::make('retribuzione_annua')
                                            ->label('Retribuzione Annua')
                                            ->numeric()
                                            ->required()
                                            ->prefix('€')
                                            ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),
                                    ])
                                    ->visible(fn (Get $get) => $get('tipo_contratto_id') == 4),

                                Select::make('corso_id')
                                    ->label('Corso')
                                    ->options(Corso::pluck('titolo', 'id'))
                                    ->searchable()
                                    ->preload()
                                    ->required()
                                    ->visible(fn (Get $get) => in_array($get('tipo_contratto_id'), [2, 3]))
                                    ->live(),

                                        Grid::make(3)
                                            ->schema([
                                        TextInput::make('cfu')
                                            ->label('CFU')
                                                    ->numeric()
                                                    ->required()
                                            ->visible(fn (Get $get) => in_array($get('tipo_contratto_id'), [2, 3])),

                                        Select::make('semestre')
                                            ->label('Semestre')
                                            ->options([
                                                '1' => 'Primo Semestre',
                                                '2' => 'Secondo Semestre'
                                            ])
                                            ->required()
                                            ->visible(fn (Get $get) => in_array($get('tipo_contratto_id'), [2, 3])),

                                        TextInput::make('anno')
                                            ->label('Anno')
                                            ->numeric()
                                            ->required()
                                            ->visible(fn (Get $get) => in_array($get('tipo_contratto_id'), [2, 3])),

                                        TextInput::make('classe')
                                            ->label('Classe')
                                            ->required()
                                            ->visible(fn (Get $get) => in_array($get('tipo_contratto_id'), [2, 3])),

                                        TextInput::make('anno_accademico')
                                            ->label('Anno Accademico')
                                            ->required()
                                            ->visible(fn (Get $get) => in_array($get('tipo_contratto_id'), [2, 3]))
                                            ->default(fn () => (date('Y')-1) . '/' . date('Y')),
                                    ])
                                    ->visible(fn (Get $get) => in_array($get('tipo_contratto_id'), [2, 3])),

                                TextInput::make('descrizione_incarico')
                                    ->label('Descrizione Incarico')
                                    ->required()
                                    ->visible(fn (Get $get) => in_array($get('tipo_contratto_id'), [2, 3, 4, 5, 6, 7])),

                                TextInput::make('ore_totali')
                                    ->label('Ore Totali')
                                    ->numeric()
                                    ->required()
                                    ->visible(fn (Get $get) => $get('tipo_contratto_id') == 1),
                            ])
                            ->collapsible(),

                        \Filament\Forms\Components\Section::make('Progetto')
                                    ->schema([
                                        Select::make('project_id')
                                            ->label('Progetto')
                                            ->relationship('project', 'title')
                                            ->searchable()
                                            ->preload()
                                            ->placeholder('Seleziona un progetto (opzionale)')
                                    ->live(),

                                        TextInput::make('ore_progetto')
                                            ->label('Ore da svolgere sul progetto')
                                            ->numeric()
                                            ->visible(fn (Get $get) => filled($get('project_id')))
                                            ->minValue(1)
                                            ->helperText('Inserisci il numero di ore da svolgere su questo progetto'),

                                        Grid::make(2)
                                            ->schema([
                                                DatePicker::make('project_data_inizio')
                                                    ->label('Data inizio attività sul progetto')
                                                    ->visible(fn (Get $get) => filled($get('project_id')))
                                                    ->minDate(fn (Get $get) => $get('data_inizio'))
                                                    ->maxDate(fn (Get $get) => $get('data_fine')),

                                                DatePicker::make('project_data_fine')
                                                    ->label('Data fine attività sul progetto')
                                                    ->visible(fn (Get $get) => filled($get('project_id')))
                                                    ->minDate(fn (Get $get) => $get('project_data_inizio') ?? $get('data_inizio'))
                                                    ->maxDate(fn (Get $get) => $get('data_fine')),
                                            ])
                                            ->visible(fn (Get $get) => filled($get('project_id'))),
                            ])
                            ->collapsible(),
                    ]),

                \Filament\Forms\Components\Group::make()
                    ->schema([
                        \Filament\Forms\Components\Section::make('Template Contratto')
                    ->schema([
                        Toggle::make('usa_template_custom')
                            ->label('Usa template personalizzato')
                            ->live(),

                        Select::make('template_id')
                            ->label('Template')
                            ->options(ContrattoTemplate::pluck('nome', 'id'))
                            ->searchable()
                            ->visible(fn (Get $get) => $get('usa_template_custom'))
                            ->live(),

                                RichEditor::make('contenuto_custom')
                            ->label('Contenuto')
                                    ->toolbarButtons([
                                        'bold',
                                        'italic',
                                        'underline',
                                        'strike',
                                        'link',
                                        'h2',
                                        'h3',
                                'paragraph',
                                'bulletList',
                                'orderedList',
                                'redo',
                                        'undo',
                            ])
                            ->visible(fn (Get $get) => $get('usa_template_custom')),

                        Section::make('Tag Disponibili')
                            ->description('Puoi utilizzare i seguenti tag nel template. Verranno sostituiti automaticamente con i valori corrispondenti.')
                            ->visible(fn (Get $get) => $get('usa_template_custom'))
                            ->schema([
                                Grid::make(3)
                                    ->schema([
                                        ViewField::make('tag_anagrafica')
                                            ->label('Anagrafica')
                                            ->view('filament.forms.components.tag-list')
                                            ->viewData([
                                                'tags' => [
                                                    '{{nome}}' => 'Nome',
                                                    '{{cognome}}' => 'Cognome',
                                                    '{{codice_fiscale}}' => 'Codice Fiscale',
                                                    '{{partita_iva}}' => 'Partita IVA',
                                                    '{{indirizzo}}' => 'Indirizzo',
                                                    '{{cap}}' => 'CAP',
                                                    '{{citta}}' => 'Città',
                                                    '{{provincia}}' => 'Provincia',
                                                ]
                                            ]),

                                        ViewField::make('tag_contratto')
                                            ->label('Contratto')
                                            ->view('filament.forms.components.tag-list')
                                            ->viewData([
                                                'tags' => [
                                                    '{{data_oggi}}' => 'Data odierna',
                                                    '{{data_inizio}}' => 'Data inizio',
                                                    '{{data_fine}}' => 'Data fine',
                                                    '{{compenso}}' => 'Compenso',
                                                    '{{ore_totali}}' => 'Ore totali',
                                                    '{{descrizione_incarico}}' => 'Descrizione incarico',
                                                ]
                                            ]),

                                        ViewField::make('tag_corso')
                                            ->label('Corso')
                                            ->view('filament.forms.components.tag-list')
                                            ->viewData([
                                                'tags' => [
                                                    '{{corso}}' => 'Nome corso',
                                                    '{{ssd}}' => 'SSD',
                                                    '{{cfu}}' => 'CFU',
                                                    '{{semestre}}' => 'Semestre',
                                                    '{{anno_accademico}}' => 'Anno accademico',
                                                ]
                                            ]),
                                    ]),
                            ]),

                        Actions::make([
                            Action::make('salva_template')
                                ->label('Salva Template')
                                ->icon('heroicon-m-plus')
                                ->color('primary')
                                ->visible(fn (Get $get) => $get('usa_template_custom'))
                                ->form([
                                    TextInput::make('nome_template')
                                        ->label('Nome Template')
                                        ->required()
                                        ->placeholder('Inserisci un nome per il template')
                                ])
                                ->modalHeading('Salva Template')
                                ->modalDescription('Inserisci un nome per salvare il template corrente')
                                ->modalSubmitActionLabel('Salva')
                                ->action(function (array $data, Get $get, Set $set) {
                                    $template = ContrattoTemplate::create([
                                        'nome' => $data['nome_template'],
                                        'descrizione' => 'Template personalizzato',
                                        'contenuto' => $get('contenuto_custom'),
                                    ]);

                                    \Illuminate\Support\Facades\Log::info('Template salvato:', [
                                        'id' => $template->id,
                                        'nome' => $template->nome,
                                        'descrizione' => $template->descrizione,
                                    ]);

                                    // Aggiorna il contratto con il nuovo template_id
                                    if ($this->contratto->exists) {
                                        $this->contratto->update([
                                            'template_id' => $template->id,
                                            'contenuto_custom' => $template->contenuto
                                        ]);
                                    }
                                    
                                    // Aggiorna i valori del form
                                    $set('template_id', $template->id);
                                    $set('contenuto_custom', $template->contenuto);
                                    $set('usa_template_custom', true);

                                    Notification::make()
                                        ->title('Template salvato')
                                        ->success()
                                        ->send();
                                }),

                            Action::make('elimina_template')
                                ->label('Elimina Template')
                                ->icon('heroicon-m-trash')
                                ->color('danger')
                                ->visible(fn (Get $get) => 
                                    $get('usa_template_custom') && 
                                    $get('template_id')
                                )
                                ->requiresConfirmation()
                                ->modalHeading('Elimina Template')
                                ->modalDescription('Sei sicuro di voler eliminare questo template? Questa azione non può essere annullata.')
                                ->modalSubmitActionLabel('Elimina')
                                ->action(function (Get $get, Set $set) {
                                    $templateId = $get('template_id');
                                    $template = ContrattoTemplate::find($templateId);
                                    
                                    if (!$template) {
                                        Notification::make()
                                            ->title('Errore')
                                            ->body('Template non trovato')
                                            ->danger()
                                            ->send();
                                        return;
                                    }

                                    // Verifica se il template è predefinito
                                    $isPredefinito = in_array($template->nome, [
                                        'Template Personale Strutturato',
                                        'Template Docenza',
                                        'Template Tutoraggio',
                                        'Template Lavoro Dipendente',
                                        'Template Collaborazione Occasionale',
                                        'Template COCOCO',
                                        'Template Collaborazione Autonoma'
                                    ]);

                                    if ($isPredefinito) {
                                        Notification::make()
                                            ->title('Errore')
                                            ->body('Non è possibile eliminare i template predefiniti')
                                            ->danger()
                                            ->send();
                                        return;
                                    }

                                    $template->delete();

                                    $set('template_id', null);
                                    $set('contenuto_custom', null);
                                    $set('usa_template_custom', false);

                                    Notification::make()
                                        ->title('Template eliminato')
                                        ->success()
                                        ->send();
                                })
                        ])
                        ->visible(fn (Get $get) => $get('usa_template_custom'))
                        ->alignment(\Filament\Support\Enums\Alignment::Right),
                    ])
                    ->persistCollapsed(false)
                    ->collapsible(),
                    ]),

                \Filament\Forms\Components\Group::make()
                    ->schema([
                        \Filament\Forms\Components\Section::make('Azioni')
                            ->schema([
                                \Filament\Forms\Components\Actions::make([
                                    \Filament\Forms\Components\Actions\Action::make('generate_pdf')
                                        ->label('Genera PDF')
                                        ->icon('heroicon-o-document')
                                        ->color('danger')
                                        ->outlined()
                                        ->visible(fn () => $this->isEditing)
                                        ->action(function () {
                                            $this->createAndSendContract();
                                        })
                                        ->requiresConfirmation()
                                        ->modalHeading('Genera PDF')
                                        ->modalDescription('Vuoi generare il PDF per questo contratto? Se esiste già un PDF, verrà sovrascritto.')
                                        ->modalSubmitActionLabel('Genera')
                                        ->modalCancelActionLabel('Annulla'),

                                    \Filament\Forms\Components\Actions\Action::make('generate_and_send')
                                        ->label('Crea contratto e invia')
                                        ->icon('heroicon-o-paper-airplane')
                                        ->color('success')
                                        ->outlined()
                                        ->visible(fn () => $this->isEditing)
                                        ->action(function () {
                                            $this->createAndSendContract();
                                            
                                            // Dopo aver generato il PDF, invialo per la firma
                                            if ($this->contratto->pdf_path) {
                                                try {
                                                    $pdfContent = Storage::disk('public')->get($this->contratto->pdf_path);
                                                    
                                                    // Crea la procedura di firma
                                                    $yousign = app('yousign');
                                                    $result = $yousign->createProcedure($pdfContent, $this->contratto->anagrafica);
                                                    
                                                    // Aggiorna il contratto con l'ID della procedura
                                                    $this->contratto->update([
                                                        'yousign_procedure_id' => $result['signature_request_id'],
                                                        'yousign_signer_id' => $result['signer_id'],
                                                        'yousign_status' => 'pending'
                                                    ]);
                                                    
                                                    \Filament\Notifications\Notification::make()
                                                        ->title('Contratto generato e inviato per la firma')
                                                        ->body('Il contratto è stato generato e inviato correttamente. Il firmatario riceverà una email con le istruzioni.')
                                                        ->success()
                                                        ->send();
                                                } catch (\Exception $e) {
                                                    \Filament\Notifications\Notification::make()
                                                        ->title('Errore')
                                                        ->body('Si è verificato un errore durante l\'invio del contratto: ' . $e->getMessage())
                                                        ->danger()
                                                        ->send();
                                                }
                                            }
                                        })
                                        ->requiresConfirmation()
                                        ->modalHeading('Genera e invia per la firma digitale')
                                        ->modalDescription('Il contratto verrà generato e inviato al firmatario tramite Yousign. Riceverà una email con le istruzioni per la firma.')
                                        ->modalSubmitActionLabel('Genera e invia')
                                        ->modalCancelActionLabel('Annulla')
                                ])
                            ])
                            ->visible(fn () => $this->isEditing)
                            ->collapsed(false),
                    ]),
            ])
            ->statePath('data')
            ->model($this->contratto);
    }
    


    public function save()
    {
        $data = $this->form->getState();
        
        \Illuminate\Support\Facades\Log::info('Form Data:', [
            'data' => $data,
            'usa_template_custom' => $data['usa_template_custom'] ?? false,
            'template_id' => $data['template_id'] ?? null,
            'corsi' => $data['corsi'] ?? [],
        ]);

        $this->validate();

        // Gestione del template personalizzato
        if ($data['usa_template_custom'] ?? false) {
            if (!empty($data['contenuto_custom'])) {
                if (empty($data['template_id'])) {
                    // Se non c'è un template selezionato, crea un nuovo template
                    $customTemplate = ContrattoTemplate::create([
                        'nome' => 'Template Personalizzato - ' . now()->format('Y-m-d H:i:s'),
                        'descrizione' => 'Template personalizzato creato manualmente',
                        'contenuto' => $data['contenuto_custom']
                    ]);
                    $data['template_id'] = $customTemplate->id;
                }
                // Mantieni sempre il contenuto custom nel contratto se è stato modificato
                $data['contenuto_custom'] = $data['contenuto_custom'];
            }
        } else {
            $data['template_id'] = null;
            $data['contenuto_custom'] = null;
        }

        // Estrai i corsi dal data array
        $corsi = $data['corsi'] ?? [];
        unset($data['corsi']);
        
        // Assicuriamoci che i campi siano nel fillable del modello
        $data = array_merge($data, [
            'descrizione_incarico' => $data['descrizione_incarico'] ?? null,
            'ore_progetto' => $data['project_id'] ? ($data['ore_progetto'] ?? null) : null,
        ]);

        DB::beginTransaction();
        try {
            if ($this->isEditing) {
                $this->contratto->update($data);
            } else {
                $this->contratto = Contratto::create($data);
            }

            // Sincronizza i corsi
            $this->contratto->corsi()->sync($corsi);

            // Crea o aggiorna l'evento associato
            $this->createOrUpdateEvent($this->contratto);

            DB::commit();

            \Filament\Notifications\Notification::make()
                ->title($this->isEditing ? 'Contratto aggiornato con successo' : 'Contratto creato con successo')
                ->success()
                ->send();

            if (!$this->isEditing) {
                $this->form->fill();
                $this->selectedAnagrafica = null;
                $this->selectedTipoContratto = null;
            }
        } catch (\Exception $e) {
            DB::rollBack();
            \Filament\Notifications\Notification::make()
                ->title('Errore durante il salvataggio')
                ->danger()
                ->send();
            throw $e;
        }
    }

    protected function createOrUpdateEvent(Contratto $contratto)
    {
        $eventTitle = "Scadenza contratto: " . $contratto->anagrafica->nome_completo;
        $eventDescription = "Scadenza del contratto di tipo: " . $contratto->tipoContratto->nome;

        Event::updateOrCreate(
            ['title' => $eventTitle],
            [
                'description' => $eventDescription,
                'start' => $contratto->data_fine,
                'end' => $contratto->data_fine->addHours(1), // Aggiungiamo un'ora per la durata dell'evento
            ]
        );
    }

    public function updatedDataUsaTemplateCustom($value)
    {
        if (!$value) {
            $this->data['template_id'] = null;
            $this->data['contenuto_custom'] = null;
        }
    }

    public function updatedDataTemplateId($value)
    {
        if ($value) {
            $template = ContrattoTemplate::find($value);
            if ($template) {
                $this->data['contenuto_custom'] = $template->contenuto;
            }
        }
    }

    public function render()
    {
        return view('livewire.create-contratto');
    }
}