<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use BezhanSalleh\FilamentShield\Traits\HasPanelShield;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Database\Eloquent\SoftDeletes;

class Anagrafica extends Model
{
    use HasFactory;
    use HasRoles;
    use HasPanelShield;
    use SoftDeletes;

    protected $table = 'anagrafiche';
    protected $guard_name = 'web'; 

    protected $fillable = [
        'is_complete', 'invited_at', 'nome', 'cognome', 'genere', 'email', 'telefono', 'invitation_token',
        'codice_fiscale', 'data_nascita', 'citta_nascita', 'provincia_nascita',
        'regione_nascita', 'indirizzo_residenza', 'numero_civico_residenza',
        'cap_residenza', 'citta_residenza', 'provincia_residenza',
        'tipologia_utente',
        'tipologia_lavoro', 'partita_iva', 'albo_professionale', 'cassa_previdenziale',
        'aliquota_inps', 'is_dipendente_pa', 'pa_denominazione', 'pa_autorizzazione_numero',
        'pa_autorizzazione_data', 'is_pensionato', 'iban',
        'has_panel_access', 'panel_email', 'panel_access_created_at',
        'is_dipendente_pa_rapporto', 'pa_denominazione_rapporto', 'pa_esclusione_tipo',
        'pa_esclusione_dettagli', 'pa_autorizzazione_numero_rapporto',
        'pa_autorizzazione_data_rapporto', 'is_pensionato_diretto', 'is_senza_previdenza',
        'tipo_pagamento', 'intestazione_conto', 'istituto_postale',
        'cv_path', 'cv_original_filename', 'cv_updated_at'
    ];

    protected $casts = [
        'has_panel_access' => 'boolean',
        'panel_access_created_at' => 'datetime',
        'cv_updated_at' => 'datetime',
        'data_nascita' => 'date',
    ];

    public function getNomeCompletoAttribute()
    {
        return "{$this->nome} {$this->cognome}";
    }

    public function contratti()
    {
        return $this->hasMany(Contratto::class);
    }

    public function isComplete(): bool
    {
        $result = !empty($this->nome) &&
           !empty($this->cognome) &&
           !empty($this->email) &&
           !empty($this->telefono) &&
           !empty($this->codice_fiscale) &&
           !empty($this->data_nascita) &&
           !empty($this->citta_nascita) &&
           !empty($this->provincia_nascita) &&
           !empty($this->regione_nascita) &&
           !empty($this->indirizzo_residenza) &&
           !empty($this->numero_civico_residenza) &&
           !empty($this->cap_residenza) &&
           !empty($this->citta_residenza) &&
           !empty($this->provincia_residenza);


        return $result;
    }

    public function tipologiaUtente()
    {
        return $this->belongsTo(TipologiaUtente::class, 'tipologia_utente', 'slug');
    }

    public function domicilio()
    {
        return $this->hasOne(Domicilio::class);
    }

    public function candidature()
    {
        return $this->hasMany(Candidature::class, 'email', 'email');
    }

    public function projects()
    {
        return $this->belongsToMany(Project::class, 'anagrafiche_project')->withTimestamps()->withPivot('work_package_id', 'budget_item_id', 'is_esterno');
    }

    protected static function boot()
    {
        parent::boot();

        // Quando un'anagrafica viene ripristinata
        static::restored(function ($anagrafica) {
            // Aggiorna i contratti correlati
            Contratto::withTrashed()
                ->where('anagrafica_id', $anagrafica->id)
                ->update(['deleted_at' => null]);
        });
    }
}
