<?php

namespace App\Observers;

use App\Models\BudgetItem;
use Filament\Notifications\Notification;

class BudgetItemObserver
{
    private function updateWorkPackageAmounts(BudgetItem $budgetItem): void
    {
        $workPackage = $budgetItem->workPackage;
        
        // Calcola il totale degli importi lordi e rendicontati per questo WP
        $totals = BudgetItem::where('work_package_id', $workPackage->id)
            ->selectRaw('SUM(gross_amount) as total_gross, SUM(reported_amount) as total_reported')
            ->first();
            
        // Aggiorna solo l'importo rendicontato del WP
        // L'importo lordo rimane quello impostato manualmente nel repeater
        $workPackage->update([
            'reported_amount' => $totals->total_reported ?? 0,
            // to_report_amount viene calcolato come virtuale nel model
        ]);
    }

    private function checkBudgetLimit(BudgetItem $budgetItem): bool
    {
        $workPackage = $budgetItem->workPackage;
        $currentTotal = $workPackage->budgetItems()
            ->where('id', '!=', $budgetItem->id)
            ->sum('gross_amount');
        
        $remainingBudget = $workPackage->gross_amount - $currentTotal;
        
        if ($budgetItem->gross_amount > $remainingBudget) {
            Notification::make()
                ->danger()
                ->title('Errore Budget')
                ->body("L'importo lordo (€ " . number_format($budgetItem->gross_amount, 2, ',', '.') . ") supera il budget rimanente del WP (€ " . number_format($remainingBudget, 2, ',', '.') . ")")
                ->send();
                
            return false;
        }
        
        return true;
    }

    public function creating(BudgetItem $budgetItem): bool
    {
        return $this->checkBudgetLimit($budgetItem);
    }

    public function updating(BudgetItem $budgetItem): bool
    {
        return $this->checkBudgetLimit($budgetItem);
    }

    public function created(BudgetItem $budgetItem): void
    {
        $this->updateWorkPackageAmounts($budgetItem);
    }

    public function updated(BudgetItem $budgetItem): void
    {
        $this->updateWorkPackageAmounts($budgetItem);
    }

    public function deleted(BudgetItem $budgetItem): void
    {
        $this->updateWorkPackageAmounts($budgetItem);
    }
} 