<?php

namespace App\Filament\Pages;

use App\Models\Contratto;
use Filament\Pages\Page;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\DeleteBulkAction;
use Illuminate\Support\Facades\Storage;
use Filament\Tables\Actions\ActionGroup;
use Filament\Forms\Components\RichEditor;
use Filament\Support\Enums\MaxWidth;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\DatePicker;
use Illuminate\Support\Facades\DB;
use App\Models\TipoContratto;
use App\Models\Event;
use Filament\Forms\Components\Section;
use Filament\Tables\Grouping\Group;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Columns\ToggleColumn;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use BezhanSalleh\FilamentShield\Traits\HasPageShield;
use Filament\Forms\Components\Grid;
use Livewire\WithFileUploads;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\ContrattoTemplate;

class GestioneContratti extends Page implements HasTable
{
    use InteractsWithTable;
    use HasPageShield;
    use WithFileUploads;

    protected static ?string $navigationGroup = 'Contratti';
    protected static ?string $navigationIcon = 'heroicon-o-document-text';
    protected static ?string $navigationLabel = 'Contratti';
    protected static ?string $title = 'Gestione Contratti';
    protected static string $view = 'filament.pages.gestione-contratti';

    protected function regeneratePdf(Contratto $contratto)
    {
        // Genera il nuovo PDF
        $pdf = $this->generatePdf($contratto);
    
        // Elimina il vecchio file PDF se esiste
        if ($contratto->pdf_path && Storage::disk('public')->exists($contratto->pdf_path)) {
            Storage::disk('public')->delete($contratto->pdf_path);
        }
    
        // Salva il nuovo PDF
        $fileName = 'contratto_' . $contratto->id . '_' . now()->format('Y-m-d_His') . '.pdf';
        $pdfPath = 'contratti/' . $fileName;
        Storage::disk('public')->put($pdfPath, $pdf);
    
        // Aggiorna il contratto con il nuovo percorso del PDF
        $contratto->update(['pdf_path' => $pdfPath]);
    
        \Filament\Notifications\Notification::make()
            ->title('PDF rigenerato con successo')
            ->success()
            ->send();
    }
    
    protected function generatePdf(Contratto $contratto)
    {
        // Carica le relazioni necessarie
        $contratto->load(['anagrafica', 'tipoContratto', 'corso']);
        $anagrafica = $contratto->anagrafica;
        $tipoContratto = $contratto->tipoContratto;
        
        // Se usa un template custom, usa il contenuto personalizzato
        if ($contratto->template_id) {
            // Prepara le variabili per i placeholder
            $variables = [
                'nome' => $anagrafica->nome,
                'cognome' => $anagrafica->cognome,
                'indirizzo' => $anagrafica->indirizzo_residenza,
                'citta' => $anagrafica->citta_residenza,
                'cap' => $anagrafica->cap_residenza,
                'provincia' => $anagrafica->provincia_residenza,
                'codice_fiscale' => $anagrafica->codice_fiscale,
                'partita_iva' => $anagrafica->partita_iva,
                'luogo_nascita' => $anagrafica->luogo_nascita,
                'data_nascita' => $anagrafica->data_nascita ? $anagrafica->data_nascita->format('d/m/Y') : '',
                'data_oggi' => now()->format('d/m/Y'),
                'compenso' => number_format($contratto->compenso, 2, ',', '.'),
                'data_inizio' => $contratto->data_inizio->format('d/m/Y'),
                'data_fine' => $contratto->data_fine->format('d/m/Y'),
                'ore_totali' => $contratto->ore_totali,
                'corso' => $contratto->corso ? $contratto->corso->titolo : '',
                'ssd' => $contratto->corso ? $contratto->corso->ssd : '',
                'cfu' => $contratto->cfu,
                'semestre' => $contratto->semestre,
                'anno_accademico' => $contratto->anno_accademico,
                'descrizione_incarico' => $contratto->descrizione_incarico,
                'numero_decreto' => $contratto->numero_decreto,
                'numero_posizioni' => $contratto->numero_posizioni,
                'area' => $contratto->area,
                'macrosettore' => $contratto->macrosettore,
                'settore_concorsuale' => $contratto->settore_concorsuale,
                'data_delibera_dipartimento' => $contratto->data_delibera_dipartimento ? $contratto->data_delibera_dipartimento->format('d/m/Y') : '',
                'data_delibera_cda' => $contratto->data_delibera_cda ? $contratto->data_delibera_cda->format('d/m/Y') : '',
                'stipendio' => number_format($contratto->stipendio, 2, ',', '.'),
                'tredicesima' => number_format($contratto->tredicesima, 2, ',', '.'),
                'regime_tempo' => $contratto->regime_tempo,
                'tipo_tempo' => $contratto->tipo_tempo,
                'tipo_orario' => $contratto->tipo_orario,
                'profilo' => $contratto->profilo,
                'livello' => $contratto->livello,
                'ore_settimanali' => $contratto->ore_settimanali,
                'retribuzione_mensile' => number_format($contratto->retribuzione_mensile, 2, ',', '.'),
                'retribuzione_annua' => number_format($contratto->retribuzione_annua, 2, ',', '.'),
            ];

            // Ottieni il contenuto del template
            $template = ContrattoTemplate::find($contratto->template_id);
            $content = $template ? $template->contenuto : '';

            // Sostituisci le variabili nel template
            foreach ($variables as $key => $value) {
                $content = str_replace('{{' . $key . '}}', $value ?? '', $content);
            }

            return PDF::loadView('pdf.contratto-custom', ['content' => $content])
                ->setPaper('a4')
                ->output();
        }

        // Mappa dei profili di contratto
        $templateMap = [
            1 => 'pdf.contratto-profilo-1',  // Personale strutturato
            2 => 'pdf.contratto-profilo-2',  // Docenza
            3 => 'pdf.contratto-profilo-3',  // Tutoraggio
            4 => 'pdf.contratto-profilo-4',  // Contratto di lavoro dipendente
            5 => 'pdf.contratto-profilo-5',  // Collaborazione occasionale
            6 => 'pdf.contratto-profilo-6',  // COCOCO
            7 => 'pdf.contratto-profilo-7',  // Collaborazione autonoma (P.IVA)
        ];

        // Se esiste un template specifico per questo tipo di contratto, usalo
        if (isset($templateMap[$tipoContratto->id])) {
            return \Barryvdh\DomPDF\Facade\Pdf::loadView($templateMap[$tipoContratto->id], [
                'contratto' => $contratto,
                'anagrafica' => $anagrafica,
                'tipoContratto' => $tipoContratto,
            ])
                ->setPaper('a4')
                ->output();
        }
        
        // Se il tipo contratto ha slug 'test', usa il template di test
        if ($tipoContratto->slug === 'test') {
            // Ottieni i dati del secondo firmatario dal database
            $secondoFirmatario = $contratto->secondoFirmatario;
            
            if (!$secondoFirmatario) {
                throw new \Exception('Secondo firmatario non specificato');
            }

            $secondoFirmatarioInfo = [
                'first_name' => $secondoFirmatario->nome,
                'last_name' => $secondoFirmatario->cognome,
                'title' => $secondoFirmatario->titolo . ' - ' . $secondoFirmatario->ruolo
            ];

            return \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.contratto-test', [
                'secondoFirmatarioInfo' => $secondoFirmatarioInfo
            ])
                ->setPaper('a4')
                ->output();
        }

        $corso = $contratto->corso;

        // Prepara le variabili per i placeholder
        $variables = [
            'nome' => $anagrafica->nome,
            'cognome' => $anagrafica->cognome,
            'indirizzo' => $anagrafica->indirizzo_residenza,
            'citta' => $anagrafica->citta_residenza,
            'data_oggi' => now()->format('d/m/Y'),
            'compenso' => number_format($contratto->compenso, 2, ',', '.'),
            'data_inizio' => $contratto->data_inizio->format('d/m/Y'),
            'data_fine' => $contratto->data_fine->format('d/m/Y'),
            'ore_totali' => $contratto->ore_totali,
            'corso' => $corso ? $corso->titolo : '',
            'ssd' => $corso ? $corso->ssd : '',
            'cfu' => $corso ? $corso->cfu : '',
            'semestre' => $corso ? $corso->semestre : '',
            'anno_accademico' => $corso ? $corso->anno_accademico : '',
        ];

        $pdf = app('dompdf.wrapper');
        $pdf->loadView('pdf.contratto', $variables);
        return $pdf->output();
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(Contratto::query()->with(['tipoContratto', 'anagrafica', 'secondoFirmatario']))
            ->recordClasses(fn (Contratto $record): string => match($record->yousign_status) {
                'completed' => 'bg-green-50 dark:bg-green-950',
                'pending' => 'bg-yellow-50 dark:bg-yellow-950',
                'rejected' => 'bg-red-50 dark:bg-red-950',
                'expired' => 'bg-gray-50 dark:bg-gray-950',
                'error' => 'bg-red-50 dark:bg-red-950',
                null => 'bg-slate-50 dark:bg-slate-950',
                default => '',
            })
            ->groups([
                Group::make('yousign_status')
                    ->label('Stato Firma')
                    ->getTitleFromRecordUsing(fn ($record) => match ($record->yousign_status) {
                        'pending' => 'In attesa di firma',
                        'completed' => 'Firmati',
                        'rejected' => 'Rifiutati',
                        'expired' => 'Scaduti',
                        'error' => 'In errore',
                        null => 'Non inviati',
                        default => ucfirst($record->yousign_status),
                    })
                    ->getKeyFromRecordUsing(fn ($record): string => $record->yousign_status ?? 'null')
                    ->collapsible(),
                Group::make('tipoContratto.nome')
                    ->label('Tipologia Contratto')
                    ->getTitleFromRecordUsing(fn ($record) => $record->tipoContratto?->nome ?? 'Nessuna tipologia')
                    ->collapsible(),
                Group::make('secondo_firmatario')
                    ->label('Firmatario')
                    ->getTitleFromRecordUsing(fn ($record) => 
                        $record->secondoFirmatario 
                            ? $record->secondoFirmatario->nome . ' ' . $record->secondoFirmatario->cognome 
                            : 'Non specificato'
                    )
                    ->getKeyFromRecordUsing(fn ($record): string => 
                        $record->secondo_firmatario ?? 'non_specificato'
                    )
                    ->collapsible(),
            ])
            ->defaultGroup('yousign_status')
            ->groupingSettingsInDropdownOnDesktop()
            ->columns([
                TextColumn::make('anagrafica.nome_completo')
                    ->label('Anagrafica')
                    ->sortable(['nome', 'cognome'])
                    ->searchable(['nome', 'cognome'])
                    ->formatStateUsing(function ($state, $record) {
                        return $record->anagrafica ? "{$record->anagrafica->nome} {$record->anagrafica->cognome}" : '[Anagrafica eliminata]';
                    })
                    ->description(fn ($record) => $record->anagrafica?->trashed() ? '(Anagrafica eliminata)' : ''),

                TextColumn::make('tipoContratto.nome_breve')
                    ->label('Tipo')
                    ->formatStateUsing(function ($state, $record) {
                        if ($record->template_id) {
                            return 'Template Custom';
                        }
                        return $state ?: 'N/A';
                    })
                    ->badge()
                    ->color(fn (?Contratto $record): string => $record->template_id ? 'warning' : ($record->tipoContratto?->colore ?? 'secondary'))
                    ->tooltip(fn (?Contratto $record): string => $record->template_id ? 'Template personalizzato' : ($record->tipoContratto?->nome ?? 'N/A')),

                TextColumn::make('secondoFirmatario.nome_completo')
                    ->label('Firmatario')
                    ->formatStateUsing(function ($state, $record) {
                        return $record->secondoFirmatario ? $state : 'Non specificato';
                    })
                    ->searchable(query: function (Builder $query, string $search): Builder {
                        return $query->whereHas('secondoFirmatario', function (Builder $query) use ($search): Builder {
                            return $query->where('nome', 'like', "%{$search}%")
                                       ->orWhere('cognome', 'like', "%{$search}%");
                        });
                    }),

                TextColumn::make('compenso')
                    ->label('Compenso')
                    ->money('eur')
                    ->sortable(),

                TextColumn::make('data_inizio')
                    ->label('Inizio')
                    ->date()
                    ->sortable(),

                TextColumn::make('data_fine')
                    ->label('Fine')
                    ->date()
                    ->sortable(),

                IconColumn::make('pdf_esistente')
                    ->label('PDF')
                    ->alignCenter()
                    ->boolean()
                    ->trueIcon(fn (Contratto $record): string => 
                        $record->is_uploaded ? 'heroicon-o-arrow-up-tray' : 'heroicon-o-document'
                    )
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor(fn (Contratto $record): string => 
                        $record->is_uploaded ? 'success' : 'primary'
                    )
                    ->falseColor('danger')
                    ->state(fn (Contratto $record): bool => $record->pdf_path !== null)
                    ->tooltip(fn (Contratto $record): string => 
                        $record->pdf_path 
                            ? ($record->is_uploaded 
                                ? 'Contratto caricato manualmente' 
                                : 'Contratto generato dal sistema')
                            : 'Contratto PDF non presente'
                    )
                    ->action(function (Contratto $record) {
                        if (!$record->pdf_path) {
                            $this->regeneratePdf($record);
                        } else {
                            return redirect()->route('download.contratto.pdf', ['id' => $record->id]);
                        }
                    }),

                IconColumn::make('contratto_firmato')
                    ->label('Firmato')
                    ->alignCenter()
                    ->boolean()
                    ->trueIcon('heroicon-o-document-check')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('gray')
                    ->state(fn (Contratto $record): bool => 
                        $record->yousign_status === 'completed' && 
                        $record->yousign_procedure_id !== null && 
                        $record->yousign_signer_id !== null
                    )
                    ->url(
                        fn (Contratto $record): ?string => 
                            $record->yousign_status === 'completed' 
                                ? route('download.contratto.firmato', ['id' => $record->id]) 
                                : null,
                        shouldOpenInNewTab: true
                    ),

                TextColumn::make('signed_at')
                    ->label('Data Firma')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->placeholder('Non firmato'),
            ])
            ->filters([
            ])
            ->actions([

                Action::make('view_anagrafica')
                ->icon('heroicon-o-user')
                ->url(fn (Contratto $record): string => Anagrafica::getUrl(['id' => $record->anagrafica_id]))
                ->openUrlInNewTab()
                ->color('success')
                ->tooltip('Visualizza l\'anagrafica di questo contratto')
                ->label('')
                ->iconButton()
                ->size('lg'),
                
                Action::make('edit')
                    ->icon('heroicon-o-pencil')
                    ->url(fn (?Contratto $record): ?string => $record ? CreateContratti::getUrl(['id' => $record->id]) : null)
                    ->color('primary')
                    ->tooltip('Modifica questo contratto')
                    ->label('')
                    ->iconButton()
                    ->size('lg'),

                Action::make('delete')
                    ->icon('heroicon-o-trash')
                    ->action(fn (?Contratto $record) => $record ? $record->delete() : null)
                    ->requiresConfirmation()
                    ->modalHeading('Elimina contratto')
                    ->modalDescription('Sei sicuro di voler eliminare questo contratto? Questa azione non può essere annullata.')
                    ->modalSubmitActionLabel('Elimina')
                    ->modalCancelActionLabel('Annulla')
                    ->color('danger')
                    ->tooltip('Elimina questo contratto')
                    ->label('')
                    ->iconButton()
                    ->size('lg'),

                ActionGroup::make([
                    Action::make('download_pdf')
                        ->icon('fas-file-pdf')
                        ->label('Scarica contratto')
                        ->url(fn (?Contratto $record): ?string => $record ? route('download.contratto.pdf', ['id' => $record->id]) : null)
                        ->openUrlInNewTab()
                        ->tooltip('Scarica contratto')
                        ->visible(fn (?Contratto $record): bool => $record !== null && $record->pdf_path !== null),

                Action::make('send_contract')
                    ->icon('heroicon-o-paper-airplane')
                    ->label('Invia contratto')
                    ->requiresConfirmation()
                    ->modalHeading('Invia per la firma digitale')
                    ->modalDescription('Il contratto verrà inviato al firmatario tramite Yousign. Riceverà una email con le istruzioni per la firma.')
                    ->modalSubmitActionLabel('Invia per la firma')
                    ->action(function (Contratto $record): void {
                        try {
                            // Leggi il contenuto del PDF
                            $pdfContent = Storage::disk('public')->get($record->pdf_path);
                            
                            // Crea la procedura di firma
                            $yousign = app('yousign');
                            $result = $yousign->createProcedure($pdfContent, $record->anagrafica);
                            
                            // Aggiorna il contratto con l'ID della procedura
                            $record->update([
                                'yousign_procedure_id' => $result['signature_request_id'],
                                'yousign_signer_id' => $result['signer_id'],
                                'yousign_status' => 'pending'
                            ]);
                            
                            \Filament\Notifications\Notification::make()
                                ->title('Contratto inviato per la firma')
                                ->body('Il contratto è stato inviato correttamente. Il firmatario riceverà una email con le istruzioni.')
                                ->success()
                                ->send();
                                
                        } catch (\Exception $e) {
                            \Filament\Notifications\Notification::make()
                                ->title('Errore')
                                ->body('Si è verificato un errore durante l\'invio del contratto: ' . $e->getMessage())
                                ->danger()
                                ->send();
                        }
                    })
                    ->visible(fn (Contratto $record): bool => 
                        $record->pdf_path !== null && 
                        $record->yousign_procedure_id === null && 
                        $record->anagrafica !== null
                    )
                    ->tooltip('Invia il contratto per la firma digitale'),

                Action::make('generate_pdf')
                    ->icon('heroicon-o-document')
                    ->label('Genera contratto')
                    ->action(function (Contratto $record) {
                        $this->regeneratePdf($record);
                    })
                    ->tooltip('Genera contratto')
                    ->visible(fn (?Contratto $record): bool => $record !== null && $record->pdf_path === null),

                Action::make('regenerate_pdf')
                    ->icon('tabler-refresh')
                    ->label('Rigenera contratto')
                    ->action(function (Contratto $record) {
                        $this->regeneratePdf($record);
                    })
                    ->requiresConfirmation()
                    ->tooltip('Rigenera contratto')
                    ->visible(fn (?Contratto $record): bool => $record !== null && $record->pdf_path !== null),

                Action::make('generate_and_send')
                    ->icon('heroicon-o-paper-airplane')
                    ->label('Genera e invia')
                    ->requiresConfirmation()
                    ->modalHeading('Genera e invia per la firma digitale')
                    ->modalDescription('Il contratto verrà generato e inviato al firmatario tramite Yousign. Riceverà una email con le istruzioni per la firma.')
                    ->modalSubmitActionLabel('Genera e invia per la firma')
                    ->action(function (Contratto $record): void {
                        try {
                            // Prima generiamo il PDF
                            $this->regeneratePdf($record);
                            
                            // Poi lo inviamo per la firma
                            $pdfContent = Storage::disk('public')->get($record->pdf_path);
                            
                            // Crea la procedura di firma
                            $yousign = app('yousign');
                            $result = $yousign->createProcedure($pdfContent, $record->anagrafica);
                            
                            // Aggiorna il contratto con l'ID della procedura
                            $record->update([
                                'yousign_procedure_id' => $result['signature_request_id'],
                                'yousign_signer_id' => $result['signer_id'],
                                'yousign_status' => 'pending'
                            ]);
                            
                            \Filament\Notifications\Notification::make()
                                ->title('Contratto generato e inviato per la firma')
                                ->body('Il contratto è stato generato e inviato correttamente. Il firmatario riceverà una email con le istruzioni.')
                                ->success()
                                ->send();
                                
                        } catch (\Exception $e) {
                            \Filament\Notifications\Notification::make()
                                ->title('Errore')
                                ->body('Si è verificato un errore durante la generazione o l\'invio del contratto: ' . $e->getMessage())
                                ->danger()
                                ->send();
                        }
                    })
                    ->visible(fn (Contratto $record): bool => 
                        $record->anagrafica !== null
                    )
                    ->tooltip('Genera il contratto e invialo per la firma digitale'),
                ])
                ->icon('heroicon-m-document-text')
                ->button()
                ->color('primary')
                ->tooltip('Azioni relative al contratto'),
            ])
            ->bulkActions([
                DeleteBulkAction::make(),
            ])
            ->headerActions([
                Action::make('create')
                    ->label('Nuovo Contratto')
                    ->icon('heroicon-o-plus')
                    ->url(fn () => CreateContratti::getUrl())
                    ->color('primary')
                    ->button(),

                Action::make('upload_contract')
                    ->label('Carica contratto firmato')
                    ->icon('heroicon-o-arrow-up-tray')
                    ->modalHeading('Carica un nuovo contratto firmato')
                    ->form([
                        Select::make('anagrafica_id')
                            ->label('Anagrafica')
                            ->options(function () {
                                return DB::table('anagrafiche')
                                    ->whereNull('deleted_at')
                                    ->where('is_complete', true)
                                    ->select('id', DB::raw("CONCAT(nome, ' ', cognome) AS nome_completo"))
                                    ->orderBy('nome')
                                    ->pluck('nome_completo', 'id');
                            })
                            ->searchable()
                            ->required(),
                        
                        Select::make('tipo_contratto_id')
                            ->label('Tipo di Contratto')
                            ->options(TipoContratto::pluck('nome', 'id'))
                            ->required()
                            ->reactive(),

                        Select::make('corsi')
                            ->label('Corsi')
                            ->multiple()
                            ->options(\App\Models\Corso::pluck('titolo', 'id'))
                            ->searchable()
                            ->preload()
                            ->visible(fn (callable $get) => $get('tipo_contratto_id') == 1),

                        TextInput::make('ore_totali')
                            ->label('Ore Totali')
                            ->numeric()
                            ->required()
                            ->visible(fn (callable $get) => $get('tipo_contratto_id') == 1),

                        TextInput::make('descrizione_incarico')
                            ->label('Descrizione Incarico')
                            ->required()
                            ->visible(fn (callable $get) => in_array($get('tipo_contratto_id'), [2, 3, 4])),

                        DatePicker::make('data_inizio')
                            ->label('Data Inizio')
                            ->required(),

                        DatePicker::make('data_fine')
                            ->label('Data Fine')
                            ->required(),

                        TextInput::make('compenso')
                            ->label('Compenso unico obbligatorio')
                            ->numeric()
                            ->required()
                            ->prefix('€'),

                        Grid::make(3)
                            ->schema([
                                TextInput::make('costo_lordo_aziendale')
                                    ->label('Costo Lordo Aziendale')
                                    ->numeric()
                                    ->prefix('€'),

                                TextInput::make('costo_lordo_dipendente')
                                    ->label('Costo Lordo Dipendente')
                                    ->numeric()
                                    ->prefix('€'),

                                TextInput::make('costo_orario')
                                    ->label('Costo Orario')
                                    ->numeric()
                                    ->prefix('€'),

                                TextInput::make('costo_giornaliero')
                                    ->label('Costo Giornaliero')
                                    ->numeric()
                                    ->prefix('€'),
                            ]),

                        Section::make('Progetto')
                            ->schema([
                                Select::make('project_id')
                                    ->label('Progetto')
                                    ->relationship('project', 'title')
                                    ->searchable()
                                    ->preload()
                                    ->placeholder('Seleziona un progetto (opzionale)')
                                    ->reactive(),

                                TextInput::make('ore_progetto')
                                    ->label('Ore da svolgere sul progetto')
                                    ->numeric()
                                    ->visible(fn (callable $get) => filled($get('project_id')))
                                    ->minValue(1)
                                    ->helperText('Inserisci il numero di ore da svolgere su questo progetto'),

                                DatePicker::make('project_data_inizio')
                                    ->label('Data inizio attività sul progetto')
                                    ->visible(fn (callable $get) => filled($get('project_id')))
                                    ->minDate(fn (callable $get) => $get('data_inizio'))
                                    ->maxDate(fn (callable $get) => $get('data_fine')),

                                DatePicker::make('project_data_fine')
                                    ->label('Data fine attività sul progetto')
                                    ->visible(fn (callable $get) => filled($get('project_id')))
                                    ->minDate(fn (callable $get) => $get('project_data_inizio') ?? $get('data_inizio'))
                                    ->maxDate(fn (callable $get) => $get('data_fine')),
                            ])
                            ->collapsible(),

                        FileUpload::make('pdf_file')
                            ->label('File PDF del contratto')
                            ->disk('local')
                            ->directory('contratti-firmati')
                            ->acceptedFileTypes(['application/pdf'])
                            ->maxSize(500 * 1024)
                            ->storeFileNamesIn('original_filename')
                            ->required(),
                    ])
                    ->action(function (array $data): void {
                        try {
                            // Crea il nuovo contratto
                            $contratto = Contratto::create([
                                'anagrafica_id' => $data['anagrafica_id'],
                                'tipo_contratto_id' => $data['tipo_contratto_id'],
                                'data_inizio' => $data['data_inizio'],
                                'data_fine' => $data['data_fine'],
                                'compenso' => $data['compenso'],
                                'costo_lordo_aziendale' => $data['costo_lordo_aziendale'] ?? null,
                                'costo_lordo_dipendente' => $data['costo_lordo_dipendente'] ?? null,
                                'costo_orario' => $data['costo_orario'] ?? null,
                                'costo_giornaliero' => $data['costo_giornaliero'] ?? null,
                                'pdf_path' => $data['pdf_file'],
                                'is_uploaded' => true,
                                'ore_totali' => $data['ore_totali'] ?? null,
                                'descrizione_incarico' => $data['descrizione_incarico'] ?? null,
                                'project_id' => $data['project_id'] ?? null,
                                'ore_progetto' => $data['ore_progetto'] ?? null,
                                'project_data_inizio' => $data['project_data_inizio'] ?? null,
                                'project_data_fine' => $data['project_data_fine'] ?? null,
                            ]);

                            // Associa i corsi se presenti
                            if (!empty($data['corsi'])) {
                                $contratto->corsi()->sync($data['corsi']);
                            }

                            // Crea l'evento per la scadenza
                            Event::create([
                                'title' => "Scadenza contratto: " . $contratto->anagrafica->nome_completo,
                                'description' => "Scadenza del contratto di tipo: " . $contratto->tipoContratto->nome,
                                'start' => $contratto->data_fine,
                                'end' => $contratto->data_fine->addHours(1),
                            ]);

                            \Filament\Notifications\Notification::make()
                                ->title('Contratto caricato con successo')
                                ->success()
                                ->send();
                        } catch (\Exception $e) {
                            \Filament\Notifications\Notification::make()
                                ->title('Errore durante il caricamento')
                                ->body($e->getMessage())
                                ->danger()
                                ->send();
                        }
                    })
                    ->button()
                    ->color('success'),
            ]);
    }

    public function getGroupedSelectableTableRecordKeys(?string $group): array
    {
        \Illuminate\Support\Facades\Log::info('Gruppo ricevuto:', ['group' => $group]);

        if ($group === null) {
            return $this->getTable()->getQuery()
                ->pluck('id')
                ->map(fn ($id): string => (string) $id)
                ->all();
        }

        $query = $this->getTable()->getQuery();

        // Gestione diretta degli stati yousign
        if (in_array($group, ['completed', 'pending', 'rejected', 'expired', 'error'])) {
            $query->where('yousign_status', $group);
        }
        // Gestione del caso "null" per i contratti non inviati
        elseif ($group === 'null') {
            $query->whereNull('yousign_status');
        }
        // Gestione dei gruppi per tipoContratto
        elseif (str_starts_with($group, 'tipoContratto.nome:')) {
            $tipoContrattoNome = str_replace('tipoContratto.nome:', '', $group);
            $query->whereHas('tipoContratto', function ($query) use ($tipoContrattoNome) {
                $query->where('nome', $tipoContrattoNome);
            });
        }

        $ids = $query->pluck('id')
            ->map(fn ($id): string => (string) $id)
            ->all();

        \Illuminate\Support\Facades\Log::info('Query SQL:', ['sql' => $query->toSql(), 'bindings' => $query->getBindings()]);
        \Illuminate\Support\Facades\Log::info('IDs selezionati:', ['ids' => $ids]);

        return $ids;
    }
}