<?php

namespace App\Filament\Pages;

use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Pages\Page;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Database\Eloquent\Model;
use App\Models\User;

class ResetPassword extends Page
{
    use InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-key';
    protected static string $view = 'filament.pages.reset-password';
    protected static bool $shouldRegisterNavigation = false;
    protected static ?string $slug = 'reset-password';

    public ?string $current_password = null;
    public ?string $password = null;
    public ?string $password_confirmation = null;

    public function mount(): void
    {
        $this->form->fill();
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('current_password')
                    ->password()
                    ->required()
                    ->label('Password Attuale'),
                TextInput::make('password')
                    ->password()
                    ->required()
                    ->minLength(8)
                    ->confirmed()
                    ->label('Nuova Password'),
                TextInput::make('password_confirmation')
                    ->password()
                    ->required()
                    ->label('Conferma Nuova Password'),
            ]);
    }

    public function submit(): void
    {
        $data = $this->form->getState();
        
        /** @var User $user */
        $user = Auth::user();

        if (!Hash::check($data['current_password'], $user->password)) {
            Notification::make()
                ->title('Password attuale non corretta')
                ->danger()
                ->send();
            return;
        }

        $user->forceFill([
            'password' => Hash::make($data['password']),
            'first_access' => false,
        ])->save();

        Notification::make()
            ->title('Password aggiornata con successo')
            ->success()
            ->send();

        Auth::logout();

        $this->redirect('/admin/login');
    }

    protected function getFormModel(): Model|string|null
    {
        return Auth::user();
    }
} 