<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ProjectResource\Pages;
use App\Filament\Resources\ProjectResource\RelationManagers;
use App\Models\Project;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use App\Models\BudgetItem;

class ProjectResource extends Resource
{
    protected static ?string $model = Project::class;
    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationLabel = 'Progetti';
    protected static ?string $navigationGroup = 'Progetti';
    protected static ?int $navigationSort = 1;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Informazioni Progetto')
                    ->schema([
                        Forms\Components\TextInput::make('title')
                            ->label('Titolo')
                            ->required()
                            ->maxLength(255),
                        Forms\Components\TextInput::make('acronym')
                            ->label('Acronimo')
                            ->maxLength(255),
                        Forms\Components\TextInput::make('cup_code')
                            ->label('Codice CUP')
                            ->maxLength(255),
                        Forms\Components\TextInput::make('codice_coge')
                            ->label('Codice COGE')
                            ->maxLength(255),
                        Forms\Components\TextInput::make('announcement')
                            ->label('Bando')
                            ->maxLength(255),
                        Forms\Components\Select::make('funding_body_id')
                            ->relationship('fundingBodies', 'name')
                            ->label('Ente Finanziatore')
                            ->multiple()
                            ->preload()
                            ->searchable(),
                        Forms\Components\TextInput::make('principal_investigator')
                            ->label('Principal Investigator')
                            ->maxLength(255),
                        Forms\Components\DatePicker::make('start_date')
                            ->label('Data Inizio')
                            ->required(),
                        Forms\Components\DatePicker::make('end_date')
                            ->label('Data Fine')
                            ->required(),
                        Forms\Components\TextInput::make('total_months')
                            ->label('Durata (mesi)')
                            ->numeric()
                            ->default(0),
                        Forms\Components\TextInput::make('iul_role')
                            ->label('Ruolo IUL')
                            ->maxLength(255),
                        Forms\Components\TextInput::make('total_project_value')
                            ->label('Valore Totale Progetto')
                            ->numeric()
                            ->prefix('€'),
                        Forms\Components\Select::make('iul_grant')
                            ->label('Contributo/GRANT IUL')
                            ->options([
                                'A.I.2.01.01' => '1) Proventi da ricerche commissionate di natura commerciale',
                                'A.I.2.01.02' => '2) Proventi da trasferimento tecnologico',
                                'A.I.2.01.03' => '3) Proventi da royalties su brevetti e altre opere d\'ingegno',
                                'A.I.2.01.04' => '4) Proventi da trasferimento tecnologico commerciali',
                                'A.I.3.01.01' => '1) Contributi per finanziamenti competitivi da Miur (PRIN, FIRB, PON) e altri',
                                'A.I.3.01.02' => '2) Contributi per finanziamenti competitivi da Unione Europea e altri soggetti (italiani',
                                'A.I.3.01.03' => '3) Contributi su progetti di ricerca da pubblico e privato',
                            ])
                            ->searchable()
                            ->placeholder('Selezionare'),
                        Forms\Components\TextInput::make('iul_cofinancing')
                            ->label('Cofinanziamento IUL')
                            ->numeric()
                            ->prefix('€'),
                    ])
                    ->columns(2),

                Forms\Components\Section::make('Work Packages')
                    ->schema([
                        Forms\Components\Repeater::make('workPackages')
                            ->relationship()
                            ->schema([
                                Forms\Components\TextInput::make('code')
                                    ->label('Codice')
                                    ->required()
                                    ->maxLength(255),
                                Forms\Components\TextInput::make('title')
                                    ->label('Titolo')
                                    ->required()
                                    ->maxLength(255),
                                Forms\Components\Textarea::make('description')
                                    ->label('Descrizione')
                                    ->columnSpanFull(),
                                Forms\Components\DatePicker::make('start_date')
                                    ->label('Data Inizio')
                                    ->required(),
                                Forms\Components\DatePicker::make('end_date')
                                    ->label('Data Fine')
                                    ->required(),
                                Forms\Components\Grid::make(3)
                                    ->schema([
                                        Forms\Components\TextInput::make('gross_amount')
                                            ->label('Importo Lordo')
                                            ->required()
                                            ->numeric()
                                            ->prefix('€')
                                            ->minValue(0),
                                        Forms\Components\TextInput::make('reported_amount')
                                            ->label('Importo Rendicontato')
                                            ->disabled()
                                            ->dehydrated(false)
                                            ->numeric()
                                            ->prefix('€')
                                            ->live()
                                            ->afterStateUpdated(function ($state, $record) {
                                                if (!$record) return;
                                                
                                                // Calcola il totale degli importi rendicontati per questo WP
                                                $totals = BudgetItem::where('work_package_id', $record->id)
                                                    ->selectRaw('SUM(reported_amount) as total_reported')
                                                    ->first();
                                                    
                                                $record->reported_amount = $totals->total_reported ?? 0;
                                                $record->save();
                                            }),
                                        Forms\Components\TextInput::make('to_report_amount')
                                            ->label('Da Rendicontare')
                                            ->disabled()
                                            ->dehydrated(false)
                                            ->numeric()
                                            ->prefix('€')
                                            ->live()
                                            ->afterStateHydrated(function ($component, $state, $record) {
                                                if (!$record) return;
                                                
                                                $component->state($record->gross_amount - $record->reported_amount);
                                            }),
                                    ]),
                            ])
                            ->columns(2)
                            ->defaultItems(0)
                            ->reorderable(false)
                            ->live()
                            ->addActionLabel('Aggiungi Work Package')
                            ->deleteAction(
                                fn (Forms\Components\Actions\Action $action) => $action->label('Rimuovi')
                            ),
                    ])
                    ->collapsible(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label('Titolo')
                    ->limit(40)
                    ->searchable(),
                // Tables\Columns\TextColumn::make('acronym')
                //     ->label('Acronimo')
                //     ->searchable(),
                Tables\Columns\TextColumn::make('cup_code')
                    ->label('Codice CUP')
                    ->searchable(),
                Tables\Columns\TextColumn::make('announcement')
                    ->label('Bando')
                    ->searchable(),
                Tables\Columns\TextColumn::make('start_date')
                    ->label('Data Inizio')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('end_date')
                    ->label('Data Fine')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('total_project_value')
                    ->label('Valore Totale')
                    ->money('EUR')
                    ->sortable(),
                // Tables\Columns\TextColumn::make('iul_grant')
                //     ->label('Contributo/GRANT IUL')
                //     ->formatStateUsing(fn (string $state): string => match ($state) {
                //         'A.I.2.01.01' => '1) Proventi da ricerche commissionate di natura commerciale',
                //         'A.I.2.01.02' => '2) Proventi da trasferimento tecnologico',
                //         'A.I.2.01.03' => '3) Proventi da royalties su brevetti e altre opere d\'ingegno',
                //         'A.I.2.01.04' => '4) Proventi da trasferimento tecnologico commerciali',
                //         'A.I.3.01.01' => '1) Contributi per finanziamenti competitivi da Miur (PRIN, FIRB, PON) e altri',
                //         'A.I.3.01.02' => '2) Contributi per finanziamenti competitivi da Unione Europea e altri soggetti (italiani',
                //         'A.I.3.01.03' => '3) Contributi su progetti di ricerca da pubblico e privato',
                //         default => $state,
                //     })
                //     ->searchable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            RelationManagers\BudgetItemsRelationManager::class,
            RelationManagers\PaymentsRelationManager::class,
            RelationManagers\DocumentsRelationManager::class,
            RelationManagers\PeopleRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProjects::route('/'),
            'create' => Pages\CreateProject::route('/create'),
            'edit' => Pages\EditProject::route('/{record}/edit'),
        ];
    }
}
