<?php

namespace App\Filament\Resources\ProjectResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;

class DocumentsRelationManager extends RelationManager
{
    protected static string $relationship = 'documents';

    protected static ?string $title = 'Documenti';

    protected static ?string $modelLabel = 'Documento';
    
    protected static ?string $pluralModelLabel = 'Documenti';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make()
                    ->schema([
                        Forms\Components\Grid::make()
                            ->schema([
                                Forms\Components\TextInput::make('description')
                                    ->label('Descrizione')
                                    ->required()
                                    ->maxLength(255)
                                    ->columnSpan(1),
                                
                                Forms\Components\Select::make('document_type')
                                    ->label('Tipo Documento')
                                    ->required()
                                    ->options([
                                        'application' => 'Application',
                                        'piano_finanziario' => 'Piano finanziario',
                                        'deliverable' => 'Deliverable',
                                        'documentazione_ricevuta' => 'Documentazione ricevuta',
                                        'documentazione_inviata' => 'Documentazione inviata',
                                        'contratti' => 'Contratti',
                                    ])
                                    ->columnSpan(1),
                                
                                Forms\Components\Select::make('status')
                                    ->label('Status')
                                    ->required()
                                    ->options([
                                        'bozza' => 'Bozza',
                                        'definitivo' => 'Definitivo',
                                    ])
                                    ->default('bozza')
                                    ->columnSpan(1),
                            ])
                            ->columns(3),

                        Forms\Components\FileUpload::make('file_path')
                            ->label('Seleziona file')
                            ->required()
                            ->directory('project-documents')
                            ->preserveFilenames()
                            ->maxSize(20480) // 20MB
                            ->acceptedFileTypes(['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'image/*'])
                            ->downloadable()
                            ->openable()
                            ->columnSpanFull()
                    ])
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('description')
                    ->label('Descrizione')
                    ->searchable()
                    ->wrap(),
                Tables\Columns\TextColumn::make('document_type')
                    ->label('Tipo Documento')
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'application' => 'Application',
                        'piano_finanziario' => 'Piano finanziario',
                        'deliverable' => 'Deliverable',
                        'documentazione_ricevuta' => 'Documentazione ricevuta',
                        'documentazione_inviata' => 'Documentazione inviata',
                        'contratti' => 'Contratti',
                        default => $state,
                    }),
                Tables\Columns\TextColumn::make('status')
                    ->label('Status')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'bozza' => 'warning',
                        'definitivo' => 'success',
                        default => 'gray',
                    }),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Data Caricamento')
                    ->dateTime('d/m/Y H:i')
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->label('Nuovo Documento'),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->label('Modifica'),
                Tables\Actions\DeleteAction::make()
                    ->label('Elimina'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
} 