<?php

namespace App\Jobs;

use App\Models\Contratto;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Throwable;

class CheckYousignStatus implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The maximum number of unhandled exceptions to allow before failing.
     *
     * @var int
     */
    public $maxExceptions = 3;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 60;

    /**
     * Delete the job if its models no longer exist.
     *
     * @var bool
     */
    public $deleteWhenMissingModels = true;

    public function __construct()
    {
        Log::info('Job CheckYousignStatus costruito');
    }

    /**
     * Dispatch the job with debug info.
     */
    public static function dispatchWithDebug()
    {
        Log::info('Dispatching CheckYousignStatus job');
        static::dispatch();
        Log::info('CheckYousignStatus job dispatched');
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        Log::info('🚀 CheckYousignStatus job started', [
            'job_id' => $this->job->getJobId(),
            'attempt' => $this->attempts()
        ]);

        try {
            // Prima vediamo tutti i contratti con procedure_id
            $allContratti = Contratto::whereNotNull('yousign_procedure_id')->get();
            Log::info('🔎 Tutti i contratti con procedure_id:', [
                'count' => $allContratti->count(),
                'contratti' => $allContratti->map(fn($c) => [
                    'id' => $c->id,
                    'procedure_id' => $c->yousign_procedure_id,
                    'status' => $c->yousign_status,
                    'signer_id' => $c->yousign_signer_id
                ])->toArray()
            ]);

            if ($allContratti->isEmpty()) {
                Log::info('Nessun contratto da controllare');
                return;
            }

            // Poi facciamo la query normale
            $contratti = Contratto::whereNotNull('yousign_procedure_id')
                ->where(function($query) {
                    $query->where('yousign_status', 'pending')
                          ->orWhereNull('yousign_status')
                          ->orWhere('yousign_status', 'processing')
                          ->orWhere('yousign_status', 'initiated')
                          ->orWhere('yousign_status', 'error');
                })
                ->get();

            Log::info('📄 Contratti da controllare dopo i filtri:', [
                'count' => $contratti->count(),
                'contratti' => $contratti->map(fn($c) => [
                    'id' => $c->id,
                    'procedure_id' => $c->yousign_procedure_id,
                    'status' => $c->yousign_status,
                    'signer_id' => $c->yousign_signer_id
                ])->toArray()
            ]);

            foreach ($contratti as $contratto) {
                Log::info('🔍 Controllo contratto:', [
                    'id' => $contratto->id,
                    'procedure_id' => $contratto->yousign_procedure_id,
                    'current_status' => $contratto->yousign_status,
                    'signer_id' => $contratto->yousign_signer_id
                ]);

                try {
                    $status = app('yousign')->getProcedureStatus($contratto->yousign_procedure_id);
                    
                    Log::info('📊 Status ricevuto da Yousign:', [
                        'contratto_id' => $contratto->id,
                        'status' => $status
                    ]);

                    $oldStatus = $contratto->yousign_status;
                    $contratto->update([
                        'yousign_status' => $status['status'],
                        'signed_at' => $status['signed_at']
                    ]);

                    Log::info('✅ Contratto aggiornato:', [
                        'id' => $contratto->id,
                        'old_status' => $oldStatus,
                        'new_status' => $status['status'],
                        'signed_at' => $status['signed_at']
                    ]);

                } catch (\Exception $e) {
                    Log::error('❌ Errore nel controllo del contratto:', [
                        'contratto_id' => $contratto->id,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    
                    // Se siamo all'ultimo tentativo, aggiorniamo lo status a error
                    if ($this->attempts() >= $this->tries) {
                        $contratto->update(['yousign_status' => 'error']);
                    }
                    
                    throw $e;
                }
            }
        } catch (\Exception $e) {
            Log::error('💥 Errore generale nel job CheckYousignStatus:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'attempt' => $this->attempts()
            ]);
            
            throw $e;
        }

        Log::info('🏁 CheckYousignStatus job completed', [
            'job_id' => $this->job->getJobId(),
            'attempt' => $this->attempts()
        ]);
    }

    /**
     * Handle a job failure.
     */
    public function failed(Throwable $exception): void
    {
        Log::error('💥 Job fallito definitivamente:', [
            'job_id' => $this->job->getJobId(),
            'error' => $exception->getMessage(),
            'trace' => $exception->getTraceAsString()
        ]);
    }
} 