<?php

namespace App\Livewire;

use App\Models\Anagrafica;
use App\Models\Comune;
use App\Models\Provincia;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use robertogallea\LaravelCodiceFiscale\CodiceFiscale;
use App\Forms\Components\GooglePlacesAutocomplete;
use Filament\Forms\Components\Section;
use Filament\Forms\Get;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Log;
use App\Jobs\SendStaffAnagraficaCompletedNotification;
use Danielebarbaro\LaravelVatEuValidator\Rules\VatNumber;
use App\Models\TipologiaUtente;

class CompleteAnagrafica extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public Anagrafica $anagrafica;
    public bool $isCompleted = false;

    public function mount($token)
    {
        $this->anagrafica = Anagrafica::where('invitation_token', $token)->firstOrFail();
    
        if (is_null($this->anagrafica->invitation_token)) {
            $this->isCompleted = true;
            Notification::make()
                ->title('Anagrafica già compilata')
                ->body('I dati di questa anagrafica sono già stati compilati.')
                ->warning()
                ->send();
        } else {
            $this->form->fill($this->anagrafica->toArray());
        }
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Dati personali')
                    ->schema([
                        TextInput::make('nome')->required(),
                        TextInput::make('cognome')->required(),
                        TextInput::make('telefono')->tel()->required(),
                        TextInput::make('email')
                            ->label('Email')
                            ->email()
                            ->required(),
                        Select::make('tipologia_utente')
                            ->label('Tipologia Utente')
                            ->options(TipologiaUtente::pluck('nome', 'slug'))
                            ->required(),
                        Select::make('genere')
                            ->label('Genere')
                            ->options([
                                'M' => 'Maschio',
                                'F' => 'Femmina',
                            ])
                            ->required(),
                        DatePicker::make('data_nascita')
                            ->label('Data di nascita')
                            ->required(),
                        TextInput::make('codice_fiscale')
                            ->label('Codice Fiscale')
                            ->required()
                            ->maxLength(16)
                            ->live()
                            ->afterStateUpdated(function ($state, callable $set, callable $get) {
                                $state = strtoupper($state); 
                                $set('codice_fiscale', $state); 
                                
                                if (is_string($state) && strlen($state) == 16) {
                                    $cf = new CodiceFiscale();
                                    if (!$cf->tryParse($state)) {
                                        $set('codice_fiscale_error', 'Il codice fiscale non è valido. ' . $cf->getError()->getMessage());
                                    } else {
                                        $set('codice_fiscale_error', null);
                                        
                                        $nome = $get('nome');
                                        $cognome = $get('cognome');
                                        $data_nascita = $get('data_nascita');
                                        $citta_nascita = $get('citta_nascita');
                                        $genere = $get('genere');
                        
                                        if ($nome && $cognome && $data_nascita && $citta_nascita && $genere) {
                                            $cfGenerato = CodiceFiscale::generate($nome, $cognome, $data_nascita, $citta_nascita, $genere);
                                            
                                            if ($state !== $cfGenerato) {
                                                $set('codice_fiscale_error', 'Il codice fiscale non corrisponde ai dati inseriti.');
                                            }
                                        }
                                    }
                                } else {
                                    $set('codice_fiscale_error', null);
                                }
                            })
                            ->helperText(fn ($get) => $get('codice_fiscale_error')),
                        Select::make('citta_nascita')
                            ->label('Città di nascita')
                            ->options(function () {
                                return Comune::orderBy('nome')
                                    ->get()
                                    ->mapWithKeys(function ($comune) {
                                        return [$comune->nome => $comune->nome_con_regione];
                                    });
                            })
                            ->searchable()
                            ->required()
                            ->live()
                            ->afterStateUpdated(function (callable $set, $state) {
                                $comune = Comune::where('nome', $state)->first();
                                if ($comune) {
                                    $provincia = Provincia::where('nome', $comune->provincia)->first();
                                    $set('provincia_nascita', $provincia ? $provincia->abbreviazione : null);
                                    $set('regione_nascita', $comune->regione);
                                }
                            }),
                        TextInput::make('provincia_nascita')
                            ->label('Provincia di nascita')
                            ->disabled()
                            ->dehydrated(true)
                            ->required(),
                        TextInput::make('regione_nascita')
                            ->label('Regione di nascita')
                            ->disabled()
                            ->dehydrated(true)
                            ->required(),
                    ])
                    ->columns(2),
                Section::make('Indirizzo di residenza')
                    ->schema([
                        GooglePlacesAutocomplete::make('indirizzo_residenza')
                            ->label('Indirizzo')
                            ->columnSpan(2)
                            ->hint("Digita l'indirizzo e clicca la voce corretta per completare l'inserimento")
                            ->required()
                            ->afterStateUpdated(function ($state, callable $set) {
                                $addressComponents = $this->getResidenzaAddressComponents($state);
                                if ($addressComponents) {
                                    $set('citta_residenza', $addressComponents['city'] ?? '');
                                    $set('cap_residenza', $addressComponents['postal_code'] ?? '');
                                    $set('provincia_residenza', $addressComponents['province'] ?? '');
                                }
                            }),
                        TextInput::make('numero_civico_residenza')
                            ->label('Numero civico')
                            ->required(),
                        TextInput::make('cap_residenza')
                            ->label('CAP')
                            ->required()
                            ->length(5),
                        TextInput::make('citta_residenza')
                            ->label('Città')
                            ->required(),
                        Select::make('provincia_residenza')
                            ->label('Provincia')
                            ->options(Provincia::pluck('nome', 'id'))
                            ->searchable()
                            ->required(),
                    ])
                    ->columns(2),
                Toggle::make('has_domicilio')
                    ->label('Di avere domicilio diverso dalla residenza')
                    ->live(),
                Section::make('Domicilio')
                    ->schema([
                        GooglePlacesAutocomplete::make('indirizzo_domicilio')
                            ->label('Indirizzo')
                            ->columnSpan(2)
                            ->hint("Digita l'indirizzo e clicca la voce corretta per completare l'inserimento")
                            ->required()
                            ->afterStateUpdated(function ($state, callable $set) {
                                $addressComponents = $this->getDomicilioAddressComponents($state);
                                if ($addressComponents) {
                                    $set('citta_domicilio', $addressComponents['city'] ?? '');
                                    $set('cap_domicilio', $addressComponents['postal_code'] ?? '');
                                    $set('provincia_domicilio', $addressComponents['province'] ?? '');
                                }
                            }),
                        TextInput::make('numero_civico_domicilio')
                            ->label('Numero civico')
                            ->required(),
                        TextInput::make('cap_domicilio')
                            ->label('CAP')
                            ->required()
                            ->length(5),
                        TextInput::make('citta_domicilio')
                            ->label('Città')
                            ->required(),
                        Select::make('provincia_domicilio')
                            ->label('Provincia')
                            ->options(Provincia::pluck('nome', 'id'))
                            ->searchable()
                            ->required(),
                    ])
                    ->columns(2)
                    ->visible(fn (Get $get) => $get('has_domicilio')),
                Section::make('Rapporto di lavoro')
                    ->schema([
                        Toggle::make('is_dipendente_pa_rapporto')
                            ->label('Di essere legato/a da un rapporto di lavoro subordinato con la Pubblica Amministrazione')
                            ->live(),
                        Toggle::make('no_esclusioni')
                            ->label('Di non rientrare nelle esclusioni soggettive previste dall\'art. 53 del D. Lgs. 165/2001 e ss.mm.ii')
                            ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto'))
                            ->live()
                            ->onColor('success')
                            ->offColor('gray'),
                        TextInput::make('pa_denominazione_rapporto')
                            ->label('Rilasciata da')
                            ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                            ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                            ->maxLength(255),
                        TextInput::make('pa_autorizzazione_numero_rapporto')
                            ->label('Autorizzazione n.')
                            ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                            ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                            ->maxLength(255),
                        DatePicker::make('pa_autorizzazione_data_rapporto')
                            ->label('del')
                            ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                            ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni')),
                        Toggle::make('pa_esclusione_tipo')
                            ->label('Di rientrare nelle esclusioni soggettive previste dall\'art. 53 del D. Lgs. 165/2001 e ss.mm.ii')
                            ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto'))
                            ->live()
                            ->onColor('danger')
                            ->offColor('gray'),
                        Select::make('pa_esclusione_dettagli')
                            ->label('Tipo di esclusione')
                            ->options([
                                'tempo_definito' => 'Regime di impegno a tempo definito o tempo parziale con prestazione lavorativa non superiore al 50% di quella a tempo pieno',
                                'formazione' => 'Art 53, comma 6 lett. f-bis, del Digs 165/01 e ss.mm.ii "attività di formazione diretta ai dipendenti della pubblica amministrazione"',
                                'altro' => 'Altro'
                            ])
                            ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('pa_esclusione_tipo'))
                            ->live()
                            ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('pa_esclusione_tipo')),
                        TextInput::make('pa_esclusione_altro')
                            ->label('Specificare altra esclusione')
                            ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('pa_esclusione_tipo') && $get('pa_esclusione_dettagli') === 'altro')
                            ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('pa_esclusione_tipo') && $get('pa_esclusione_dettagli') === 'altro')
                            ->maxLength(500),
                        Toggle::make('is_pensionato_diretto')
                            ->label('Di essere titolare di pensione diretta')
                            ->live(),
                        Toggle::make('is_senza_previdenza')
                            ->label('Non iscritto ad alcuna forma previdenziale')
                            ->onIcon('heroicon-m-user')
                            ->offIcon('heroicon-m-user'),
                    ]),
                Section::make('Dati fiscali e previdenziali')
                    ->schema([
                        Select::make('tipologia_lavoro')
                            ->label('Tipologia di lavoro')
                            ->options([
                                'lavoro_autonomo_professionale' => 'Lavoro autonomo professionale abituale',
                                'regime_fiscale_vantaggio' => 'Contribuente nel nuovo regime fiscale di vantaggio',
                                'lavoro_autonomo_occasionale' => 'Lavoro autonomo occasionale'
                            ])
                            ->required()
                            ->live(),
                        TextInput::make('partita_iva')
                            ->label('Partita IVA')
                            ->visible(fn (Get $get) => $get('tipologia_lavoro') === 'lavoro_autonomo_professionale')
                            ->rules(['required_if:tipologia_lavoro,lavoro_autonomo_professionale'])
                            ->rules(function ($get, $state) {
                                if (preg_match('/^[A-Z]{2}/i', $state)) {
                                    return [new VatNumber(['format_only' => true])];
                                }
                                return ['regex:/^[0-9]{11}$/'];
                            })
                            ->live()
                            ->hint('Se si inserisce il prefisso del paese (es. IT12345678901) verrà effettuata la verifica nel database VIES')
                            ->afterStateUpdated(function ($state, callable $set) {
                                if (!empty($state)) {
                                    if (preg_match('/^[A-Z]{2}/i', $state)) {
                                        try {
                                            $validator = new \Danielebarbaro\LaravelVatEuValidator\VatValidator();
                                            $isValid = $validator->validate($state);
                                            if ($isValid) {
                                                $set('partita_iva_message', 'Partita IVA verificata nel database VIES ✓');
                                                $set('partita_iva_message_color', 'success');
                                            } else {
                                                $set('partita_iva_message', 'Partita IVA non presente nel database VIES');
                                                $set('partita_iva_message_color', 'warning');
                                            }
                                        } catch (\Exception $e) {
                                            $set('partita_iva_message', null);
                                        }
                                    } else {
                                        $set('partita_iva_message', null);
                                    }
                                } else {
                                    $set('partita_iva_message', null);
                                }
                            })
                            ->helperText(fn (Get $get) => $get('partita_iva_message'))
                            ->hintColor(fn (Get $get) => $get('partita_iva_message_color')),
                        TextInput::make('cassa_previdenziale')
                            ->label('Cassa o ente previdenziale')
                            ->visible(fn (Get $get) => $get('tipologia_lavoro') === 'lavoro_autonomo_professionale'),
                        TextInput::make('albo_professionale')
                            ->label('Albo o elenco professionale')
                            ->visible(fn (Get $get) => $get('tipologia_lavoro') === 'lavoro_autonomo_professionale'),
                        Select::make('aliquota_inps')
                            ->label('Aliquota INPS')
                            ->options([
                                '33.72' => '33,72% (non iscritti ad altre forme pensionistiche)',
                                '24.00' => '24,00% (pensionati o iscritti ad altra forma pensionistica)',
                                '24.00_altra' => '24,00% (già iscritto ad altra cassa previdenziale)'
                            ])
                            ->visible(fn (Get $get) => $get('tipologia_lavoro') === 'lavoro_autonomo_occasionale'),
                    ]),
                Section::make('Dati pagamento')
                    ->schema([
                        Select::make('tipo_pagamento')
                            ->label('Tipo di pagamento')
                            ->options([
                                'bancario' => 'Accredito su c/c bancario',
                                'postale' => 'Accredito su c/c postale'
                            ])
                            ->required()
                            ->live(),
                        TextInput::make('intestazione_conto')
                            ->label('c/c intestato a')
                            ->required()
                            ->visible(fn (Get $get) => $get('tipo_pagamento') === 'bancario'),
                        TextInput::make('iban')
                            ->label('IBAN')
                            ->required()
                            ->visible(fn (Get $get) => $get('tipo_pagamento') === 'bancario')
                            ->hint('Il conto corrente deve essere intestato o cointestato alla persona che ha ricevuto l\'incarico')
                            ->rules(['iban']),
                        TextInput::make('istituto_postale')
                            ->label('Istituto postale')
                            ->required()
                            ->visible(fn (Get $get) => $get('tipo_pagamento') === 'postale'),
                    ]),
            ])
            ->statePath('data');
    }

    public function save()
    {
        $data = $this->form->getState();

         // Validazione del codice fiscale
        $cfGenerato = CodiceFiscale::generate(
            $data['nome'],
            $data['cognome'],
            $data['data_nascita'],
            $data['citta_nascita'],
            $data['genere']
        );

        if (strtoupper($data['codice_fiscale']) !== $cfGenerato) {
            Notification::make()
                ->title('Errore di validazione')
                ->body('Il codice fiscale non corrisponde ai dati inseriti.')
                ->danger()
                ->send();
            return;
        }

        $this->anagrafica->update($data);
        $this->isCompleted = true;
        $this->anagrafica->invitation_token = null;
        $this->anagrafica->is_complete = $this->anagrafica->isComplete(); 
        $this->anagrafica->save();
    
        Log::info('Anagrafica updated', [
            'id' => $this->anagrafica->id,
            'is_complete' => $this->anagrafica->is_complete
        ]);
    
        // Dispatch del job per inviare l'email in modo asincrono
        SendStaffAnagraficaCompletedNotification::dispatch($this->anagrafica);
    
        Notification::make()
            ->title('Anagrafica completata con successo')
            ->success()
            ->send();
    
    }

    public function render()
    {
        return view('livewire.complete-anagrafica', [
            'isCompleted' => $this->isCompleted,
        ]);
    }

    protected function getResidenzaAddressComponents($placeId)
    {
        try {
            $client = new \GuzzleHttp\Client();
            $response = $client->get("https://maps.googleapis.com/maps/api/place/details/json", [
                'query' => [
                    'place_id' => $placeId,
                    'key' => config('services.google.places_key'),
                    'language' => 'it'
                ]
            ]);

            $place = json_decode($response->getBody(), true);

            if (!isset($place['result']['address_components'])) {
                return null;
            }

            $components = [];
            foreach ($place['result']['address_components'] as $component) {
                if (in_array('locality', $component['types'])) {
                    $components['city'] = $component['long_name'];
                }
                if (in_array('postal_code', $component['types'])) {
                    $components['postal_code'] = $component['long_name'];
                }
                if (in_array('administrative_area_level_2', $component['types'])) {
                    $components['province'] = $component['short_name'];
                }
            }

            return $components;
        } catch (\Exception $e) {
            return null;
        }
    }

    protected function getDomicilioAddressComponents($placeId)
    {
        try {
            $client = new \GuzzleHttp\Client();
            $response = $client->get("https://maps.googleapis.com/maps/api/place/details/json", [
                'query' => [
                    'place_id' => $placeId,
                    'key' => config('services.google.places_key'),
                    'language' => 'it'
                ]
            ]);

            $place = json_decode($response->getBody(), true);

            if (!isset($place['result']['address_components'])) {
                return null;
            }

            $components = [];
            foreach ($place['result']['address_components'] as $component) {
                if (in_array('locality', $component['types'])) {
                    $components['city'] = $component['long_name'];
                }
                if (in_array('postal_code', $component['types'])) {
                    $components['postal_code'] = $component['long_name'];
                }
                if (in_array('administrative_area_level_2', $component['types'])) {
                    $components['province'] = $component['short_name'];
                }
            }

            return $components;
        } catch (\Exception $e) {
            return null;
        }
    }
}