<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class YousignServiceProvider extends ServiceProvider
{
    public function register()
    {
        $this->app->singleton('yousign', function ($app) {
            return new class {
                protected $baseUrl;
                protected $apiKey;

                public function __construct()
                {
                    $this->baseUrl = config('services.yousign.url');
                    $this->apiKey = config('services.yousign.key');

                    if (empty($this->apiKey)) {
                        throw new \Exception('Yousign API key non configurata. Controlla il file .env');
                    }
                }

                protected function getHeaders()
                {
                    return [
                        'Authorization' => sprintf('Bearer %s', trim($this->apiKey))
                    ];
                }

                protected function calculateSignatureCoordinates($file)
                {
                    try {
                        // Inizializziamo il parser
                        $parser = new \Smalot\PdfParser\Parser();
                        $pdf = $parser->parseContent($file);
                        
                        // Otteniamo i dati di posizione del testo della prima pagina
                        $page = $pdf->getPages()[0];
                        $dataTm = $page->getDataTm();
                        
                        Log::info('📄 Dati posizione testo:', ['dataTm' => $dataTm]);

                        // Troviamo le posizioni di Francesco Mugnai e Dott. Mario Rossi
                        $firstSignerPos = null;
                        $directorPos = null;

                        // Altezza pagina A4 in punti
                        $pageHeight = 842;

                        foreach ($dataTm as $textData) {
                            $text = $textData[1];
                            $position = $textData[0];
                            
                            if (str_contains($text, 'Francesco Mugnai')) {
                                $firstSignerPos = [
                                    'x' => (int)($position[4]),      // Stessa X del nome
                                    'y' => (int)($pageHeight - $position[5] + 20)  // 20 punti sotto il nome
                                ];
                            }
                            
                            if (str_contains($text, 'Dott. Mario Rossi')) {
                                $directorPos = [
                                    'x' => (int)($position[4]),      // Stessa X del nome
                                    'y' => (int)($pageHeight - $position[5] + 20)  // 20 punti sotto il nome
                                ];
                            }
                        }

                        // Se non troviamo le posizioni, usiamo dei valori di fallback
                        $firstSignerPos = $firstSignerPos ?? ['x' => 64, 'y' => 267];  // 842 - 595 + 20 = 267
                        $directorPos = $directorPos ?? ['x' => 64, 'y' => 373];        // 842 - 489 + 20 = 373

                        Log::info('📏 Coordinate calcolate:', [
                            'first_signer' => $firstSignerPos,
                            'director' => $directorPos
                        ]);
                        
                        return [
                            'first_signer' => [
                                'page' => 1,
                                'x' => $firstSignerPos['x'],
                                'y' => $firstSignerPos['y']
                            ],
                            'director' => [
                                'page' => 1,
                                'x' => $directorPos['x'],
                                'y' => $directorPos['y']
                            ]
                        ];
                    } catch (\Exception $e) {
                        Log::error('❌ Errore nel calcolo delle coordinate: ' . $e->getMessage());
                        
                        // Coordinate di fallback in caso di errore
                        return [
                            'first_signer' => [
                                'page' => 1,
                                'x' => 64,
                                'y' => 267  // 842 - 595 + 20 = 267
                            ],
                            'director' => [
                                'page' => 1,
                                'x' => 64,
                                'y' => 373  // 842 - 489 + 20 = 373
                            ]
                        ];
                    }
                }

                public function createProcedure($file, $signer, $secondoFirmatario = 'direttore')
                {
                    // Step 1: Creiamo una richiesta di firma
                    $signatureRequest = Http::withHeaders($this->getHeaders())
                        ->withBody(json_encode([
                            'name' => 'Firma Contratto',
                            'delivery_mode' => 'email',
                            'timezone' => 'Europe/Rome',
                            'ordered_signers' => true,
                            'custom_experience_id' => '75496840-f1cc-4ef9-a984-ea33b4d10e61'
                        ]), 'application/json')
                        ->post($this->baseUrl . '/signature_requests');

                    if (!$signatureRequest->successful()) {
                        throw new \Exception('Errore nella creazione della richiesta di firma: ' . $signatureRequest->body());
                    }

                    $signatureRequestId = $signatureRequest->json()['id'];

                    // Step 2: Upload document
                    $documentUpload = Http::withHeaders($this->getHeaders())
                        ->attach(
                            'file', 
                            $file, 
                            'document.pdf', 
                            ['Content-Type' => 'application/pdf']
                        )
                        ->post($this->baseUrl . '/signature_requests/' . $signatureRequestId . '/documents', [
                            'nature' => 'signable_document'
                        ]);

                    if (!$documentUpload->successful()) {
                        throw new \Exception('Errore nel caricamento del documento: ' . $documentUpload->body());
                    }

                    $documentId = $documentUpload->json()['id'];

                    // Calcoliamo le coordinate delle firme
                    $coordinates = $this->calculateSignatureCoordinates($file);

                    // Step 3: Aggiungiamo il primo firmatario (utente)
                    $phoneNumber = $signer->telefono;
                    if (!str_starts_with($phoneNumber, '+')) {
                        $phoneNumber = '+39' . preg_replace('/[^0-9]/', '', $phoneNumber);
                    }

                    $signerRequest = Http::withHeaders($this->getHeaders())
                        ->withBody(json_encode([
                            'info' => [
                                'first_name' => $signer->nome,
                                'last_name' => $signer->cognome,
                                'email' => $signer->email,
                                'phone_number' => $phoneNumber,
                                'locale' => 'it'
                            ],
                            'signature_authentication_mode' => 'no_otp',
                            'signature_level' => 'electronic_signature',
                            'fields' => [
                                [
                                    'document_id' => $documentId,
                                    'type' => 'signature',
                                    'height' => 40,
                                    'width' => 85,
                                    'page' => $coordinates['first_signer']['page'],
                                    'x' => $coordinates['first_signer']['x'],
                                    'y' => $coordinates['first_signer']['y']
                                ]
                            ]
                        ]), 'application/json')
                        ->post($this->baseUrl . '/signature_requests/' . $signatureRequestId . '/signers');

                    if (!$signerRequest->successful()) {
                        throw new \Exception('Errore nell\'aggiunta del firmatario: ' . $signerRequest->body());
                    }

                    $signerId = $signerRequest->json()['id'];

                    // Step 4: Aggiungiamo il secondo firmatario in base alla selezione
                    $secondiFirmatari = [
                        'direttore' => [
                            'first_name' => 'Mario',
                            'last_name' => 'Rossi',
                            'email' => 'f.mugnai@indire.it',
                            'title' => 'Direttore Generale'
                        ],
                        'presidente' => [
                            'first_name' => 'Massimiliano',
                            'last_name' => 'Bizzocchi',
                            'email' => 'f.mugnai@indire.it',
                            'title' => 'Presidente'
                        ],
                        'rettore' => [
                            'first_name' => 'Carlo',
                            'last_name' => 'Tasso',
                            'email' => 'f.mugnai@indire.it',
                            'title' => 'Rettore'
                        ]
                    ];

                    $secondoFirmatarioInfo = $secondiFirmatari[$secondoFirmatario] ?? $secondiFirmatari['direttore'];

                    $directorRequest = Http::withHeaders($this->getHeaders())
                        ->withBody(json_encode([
                            'info' => [
                                'first_name' => $secondoFirmatarioInfo['first_name'],
                                'last_name' => $secondoFirmatarioInfo['last_name'],
                                'email' => $secondoFirmatarioInfo['email'],
                                'locale' => 'it'
                            ],
                            'signature_authentication_mode' => 'no_otp',
                            'signature_level' => 'electronic_signature',
                            'fields' => [
                                [
                                    'document_id' => $documentId,
                                    'type' => 'signature',
                                    'height' => 40,
                                    'width' => 85,
                                    'page' => $coordinates['director']['page'],
                                    'x' => $coordinates['director']['x'],
                                    'y' => $coordinates['director']['y']
                                ]
                            ]
                        ]), 'application/json')
                        ->post($this->baseUrl . '/signature_requests/' . $signatureRequestId . '/signers');

                    if (!$directorRequest->successful()) {
                        throw new \Exception('Errore nell\'aggiunta del secondo firmatario: ' . $directorRequest->body());
                    }

                    $directorId = $directorRequest->json()['id'];

                    // Step 5: Activate the signature request
                    $activateRequest = Http::withHeaders($this->getHeaders())
                        ->withBody('{}', 'application/json')
                        ->post($this->baseUrl . '/signature_requests/' . $signatureRequestId . '/activate');

                    if (!$activateRequest->successful()) {
                        throw new \Exception('Errore nell\'attivazione della richiesta: ' . $activateRequest->body());
                    }

                    return [
                        'signature_request_id' => $signatureRequestId,
                        'signer_id' => $signerId,
                        'director_id' => $directorId,
                        'document_id' => $documentId
                    ];
                }

                public function getSignedPdf($signatureRequestId)
                {
                    // Scarica tutti i documenti firmati della procedura
                    $response = Http::withHeaders($this->getHeaders())
                        ->get($this->baseUrl . '/signature_requests/' . $signatureRequestId . '/documents/download');

                    if (!$response->successful()) {
                        throw new \Exception('Errore nel download del documento firmato: ' . $response->body());
                    }

                    return $response->body();
                }

                public function getProcedureStatus($signatureRequestId)
                {
                    Log::info('🔄 Controllo stato firma per signatureRequestId: ' . $signatureRequestId);

                    // Prima otteniamo i dettagli della signature request
                    $response = Http::withHeaders($this->getHeaders())
                        ->get($this->baseUrl . '/signature_requests/' . $signatureRequestId);

                    if (!$response->successful()) {
                        Log::error('❌ Errore nel recupero dello stato:', [
                            'status' => $response->status(),
                            'body' => $response->body(),
                            'headers' => $response->headers()
                        ]);
                        throw new \Exception('Errore nel recupero dello stato: ' . $response->body());
                    }

                    $data = $response->json();
                    Log::info('📥 Risposta signature request:', $data);
                    
                    $signer = $data['signers'][0] ?? null;
                    
                    if (!$signer) {
                        Log::error('❌ Nessun firmatario trovato per la signature request: ' . $signatureRequestId);
                        throw new \Exception('Nessun firmatario trovato');
                    }

                    $signerId = $signer['id'];
                    $signerStatus = $signer['status'];

                    Log::info('👤 Stato firmatario:', [
                        'signerId' => $signerId,
                        'status' => $signerStatus,
                        'email' => $signer['info']['email'] ?? null,
                        'name' => ($signer['info']['first_name'] ?? '') . ' ' . ($signer['info']['last_name'] ?? '')
                    ]);

                    // Se lo stato è "done", verifichiamo l'audit trail
                    if ($signerStatus === 'done') {
                        try {
                            // Chiamiamo l'endpoint corretto dell'audit trail per il firmatario specifico
                            $auditTrail = Http::withHeaders($this->getHeaders())
                                ->get($this->baseUrl . '/signature_requests/' . $signatureRequestId . '/signers/' . $signerId . '/audit_trails');
                            
                            if ($auditTrail->successful()) {
                                $auditTrailData = $auditTrail->json();
                                Log::info('📋 Audit trail recuperato:', $auditTrailData);

                                // Verifichiamo anche il PDF dell'audit trail
                                try {
                                    $auditTrailPdf = Http::withHeaders($this->getHeaders())
                                        ->get($this->baseUrl . '/signature_requests/' . $signatureRequestId . '/signers/' . $signerId . '/audit_trails/download');
                                    
                                    if ($auditTrailPdf->successful()) {
                                        Log::info('📑 PDF Audit trail disponibile');
                                    }
                                } catch (\Exception $e) {
                                    Log::warning('⚠️ PDF Audit trail non disponibile: ' . $e->getMessage());
                                }

                                return [
                                    'status' => 'completed',
                                    'signed_at' => now(),
                                    'signature_link' => $signer['signature_link'] ?? null,
                                    'raw_response' => $data,
                                    'audit_trail' => $auditTrailData
                                ];
                            } else {
                                Log::warning('⚠️ Audit trail non disponibile:', [
                                    'status' => $auditTrail->status(),
                                    'response' => $auditTrail->json()
                                ]);
                            }
                        } catch (\Exception $e) {
                            Log::error('❌ Errore nel recupero dell\'audit trail: ' . $e->getMessage(), [
                                'trace' => $e->getTraceAsString()
                            ]);
                        }
                    }

                    // Mappiamo gli stati di Yousign agli stati del nostro sistema
                    $status = match ($signerStatus) {
                        'initiated' => 'pending',    // Il firmatario ha ricevuto il link ma non ha ancora firmato
                        'processing' => 'pending',   // Il firmatario sta firmando
                        'signed' => 'completed',     // Il firmatario ha firmato
                        'done' => 'completed',       // Il firmatario ha firmato (stato alternativo)
                        'rejected' => 'rejected',    // Il firmatario ha rifiutato
                        'expired' => 'expired',      // Il link è scaduto
                        default => 'pending'
                    };

                    Log::info('📊 Stato finale:', [
                        'signatureRequestId' => $signatureRequestId,
                        'signerId' => $signerId,
                        'status' => $status,
                        'raw_status' => $signerStatus
                    ]);

                    return [
                        'status' => $status,
                        'signed_at' => ($status === 'completed' || $signerStatus === 'signed' || $signerStatus === 'done') ? now() : null,
                        'signature_link' => $signer['signature_link'] ?? null,
                        'raw_response' => $data
                    ];
                }

                public function downloadSignedDocument($signatureRequestId)
                {
                    Log::info('📥 Download documento firmato per signatureRequestId: ' . $signatureRequestId);

                    // Prima otteniamo i dettagli della signature request per avere l'ID del documento
                    $response = Http::withHeaders($this->getHeaders())
                        ->get($this->baseUrl . '/signature_requests/' . $signatureRequestId);

                    if (!$response->successful()) {
                        Log::error('❌ Errore nel recupero dei dettagli:', [
                            'status' => $response->status(),
                            'body' => $response->body()
                        ]);
                        throw new \Exception('Errore nel recupero dei dettagli: ' . $response->body());
                    }

                    $data = $response->json();
                    $documentId = $data['documents'][0]['id'] ?? null;

                    if (!$documentId) {
                        throw new \Exception('Nessun documento trovato nella signature request');
                    }

                    // Ora scarichiamo il documento firmato
                    $download = Http::withHeaders($this->getHeaders())
                        ->withOptions([
                            'decode_content' => false,
                        ])
                        ->get($this->baseUrl . '/signature_requests/' . $signatureRequestId . '/documents/' . $documentId . '/download');

                    if (!$download->successful()) {
                        Log::error('❌ Errore nel download del documento firmato:', [
                            'status' => $download->status(),
                            'body' => $download->body()
                        ]);
                        throw new \Exception('Errore nel download del documento firmato: ' . $download->body());
                    }

                    Log::info('✅ Documento firmato scaricato con successo');
                    
                    // Restituiamo il contenuto binario del PDF
                    return $download->body();
                }

                public function boot()
                {
                    //
                }
            };
        });
    }

    public function boot()
    {
        //
    }
} 